<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // 1) Spatie Permissions (RBAC)
        if (!Schema::hasTable('permissions')) {
            Schema::create('permissions', function (Blueprint $table) {
                $table->bigIncrements('id');
                $table->string('name');
                $table->string('guard_name')->default('web');
                $table->timestamps();
            });
        }

        if (!Schema::hasTable('roles')) {
            Schema::create('roles', function (Blueprint $table) {
                $table->bigIncrements('id');
                $table->string('name');
                $table->string('guard_name')->default('web');
                $table->timestamps();
            });
        }

        if (!Schema::hasTable('model_has_permissions')) {
            Schema::create('model_has_permissions', function (Blueprint $table) {
                $table->unsignedBigInteger('permission_id');
                $table->string('model_type');
                $table->unsignedBigInteger('model_id');
                $table->index(['model_id', 'model_type'], 'model_has_permissions_model_id_model_type_index');
                $table->foreign('permission_id')->references('id')->on('permissions')->onDelete('cascade');
                $table->primary(['permission_id', 'model_id', 'model_type'], 'model_has_permissions_permission_model_type_primary');
            });
        }

        if (!Schema::hasTable('model_has_roles')) {
            Schema::create('model_has_roles', function (Blueprint $table) {
                $table->unsignedBigInteger('role_id');
                $table->string('model_type');
                $table->unsignedBigInteger('model_id');
                $table->index(['model_id', 'model_type'], 'model_has_roles_model_id_model_type_index');
                $table->foreign('role_id')->references('id')->on('roles')->onDelete('cascade');
                $table->primary(['role_id', 'model_id', 'model_type'], 'model_has_roles_role_model_type_primary');
            });
        }

        if (!Schema::hasTable('role_has_permissions')) {
            Schema::create('role_has_permissions', function (Blueprint $table) {
                $table->unsignedBigInteger('permission_id');
                $table->unsignedBigInteger('role_id');
                $table->foreign('permission_id')->references('id')->on('permissions')->onDelete('cascade');
                $table->foreign('role_id')->references('id')->on('roles')->onDelete('cascade');
                $table->primary(['permission_id', 'role_id'], 'role_has_permissions_permission_id_role_id_primary');
            });
        }

        // 2) Settings
        if (!Schema::hasTable('settings')) {
            Schema::create('settings', function (Blueprint $table) {
                $table->id();
                $table->string('key', 150)->unique();
                $table->text('value')->nullable();
                $table->string('type', 20)->default('string');
                $table->string('group', 50)->nullable();
                $table->text('description')->nullable();
                $table->boolean('autoload')->default(true);
                $table->timestamps();
            });
        }

        // 3) Contact messages
        if (!Schema::hasTable('contact_messages')) {
            Schema::create('contact_messages', function (Blueprint $table) {
                $table->id();
                $table->string('name');
                $table->string('email');
                $table->string('phone')->nullable();
                $table->string('subject')->nullable();
                $table->text('message');
                $table->string('ip', 45)->nullable();
                $table->string('user_agent')->nullable();
                $table->timestamps();
            });
        }

        // 4) Order status events
        if (!Schema::hasTable('order_status_events')) {
            Schema::create('order_status_events', function (Blueprint $table) {
                $table->id();
                // Use unsigned integers and indexes to avoid FK type mismatches with legacy INT PKs
                $table->unsignedInteger('order_id');
                $table->string('status', 50);
                $table->unsignedInteger('user_id')->nullable();
                $table->string('note', 255)->nullable();
                $table->timestamps();
                $table->index('order_id');
                $table->index('user_id');
            });
        }

        // 5) Inventory movements + stock on dishes
        if (!Schema::hasColumn('dishes', 'stock')) {
            Schema::table('dishes', function (Blueprint $table) {
                $table->integer('stock')->default(0)->after('price');
            });
        }
        if (!Schema::hasTable('inventory_movements')) {
            Schema::create('inventory_movements', function (Blueprint $table) {
                $table->id();
                $table->unsignedInteger('dish_id');
                $table->integer('change');
                $table->string('reason')->nullable();
                $table->unsignedInteger('order_id')->nullable();
                $table->unsignedInteger('user_id')->nullable();
                $table->text('note')->nullable();
                $table->timestamps();
                $table->index('dish_id');
                $table->index('order_id');
                $table->index('user_id');
            });
        }

        // 6) Order financials & payments
        if (!Schema::hasColumn('orders', 'service_charge')) {
            Schema::table('orders', function (Blueprint $table) {
                $table->decimal('service_charge', 10, 2)->default(0)->after('discount');
            });
        }
        if (!Schema::hasColumn('orders', 'tip')) {
            Schema::table('orders', function (Blueprint $table) {
                $table->decimal('tip', 10, 2)->default(0)->after('total');
            });
        }
        if (!Schema::hasColumn('orders', 'payment_reference')) {
            Schema::table('orders', function (Blueprint $table) {
                $table->string('payment_reference', 100)->nullable()->after('payment_method');
            });
        }
        if (!Schema::hasColumn('orders', 'payment_verified')) {
            Schema::table('orders', function (Blueprint $table) {
                $table->boolean('payment_verified')->default(false)->after('payment_reference');
            });
        }

        // 7) Users hardening
        if (!Schema::hasColumn('users', 'phone_verified')) {
            Schema::table('users', function (Blueprint $table) {
                $table->boolean('phone_verified')->default(false)->after('role');
            });
        }

        // 8) Infra tables (optional but useful)
        if (!Schema::hasTable('cache')) {
            Schema::create('cache', function (Blueprint $table) {
                $table->string('key')->primary();
                $table->mediumText('value');
                $table->integer('expiration');
            });
        }
        if (!Schema::hasTable('jobs')) {
            Schema::create('jobs', function (Blueprint $table) {
                $table->bigIncrements('id');
                $table->string('queue')->index();
                $table->longText('payload');
                $table->unsignedTinyInteger('attempts');
                $table->unsignedInteger('reserved_at')->nullable();
                $table->unsignedInteger('available_at');
                $table->unsignedInteger('created_at');
            });
        }
    }

    public function down(): void
    {
        // Drop optional infra tables
        Schema::dropIfExists('jobs');
        Schema::dropIfExists('cache');

        // Revert user/order columns cautiously
        if (Schema::hasColumn('users', 'phone_verified')) {
            Schema::table('users', function (Blueprint $table) { $table->dropColumn('phone_verified'); });
        }
        foreach (['payment_verified','payment_reference','tip','service_charge'] as $col) {
            if (Schema::hasColumn('orders', $col)) {
                Schema::table('orders', function (Blueprint $table) use ($col) { $table->dropColumn($col); });
            }
        }
        if (Schema::hasColumn('dishes', 'stock')) {
            Schema::table('dishes', function (Blueprint $table) { $table->dropColumn('stock'); });
        }

        // Drop add-on feature tables
        Schema::dropIfExists('inventory_movements');
        Schema::dropIfExists('order_status_events');
        Schema::dropIfExists('contact_messages');
        Schema::dropIfExists('settings');

        // Drop RBAC tables (in reverse FK order)
        Schema::dropIfExists('role_has_permissions');
        Schema::dropIfExists('model_has_roles');
        Schema::dropIfExists('model_has_permissions');
        Schema::dropIfExists('roles');
        Schema::dropIfExists('permissions');
    }
};
