<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        if (!Schema::hasTable('stays')) {
            Schema::create('stays', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->constrained('users')->cascadeOnDelete();
                $table->string('room_number', 50);
                $table->dateTime('check_in_at');
                $table->dateTime('check_out_at')->nullable();
                $table->string('status', 20)->default('active'); // active|closed|suspended
                $table->string('pms_ref', 100)->nullable();
                $table->timestamps();
                $table->index(['user_id','status']);
            });
        }

        // Extend orders table with new delivery + verification fields (idempotent guards)
        Schema::table('orders', function (Blueprint $table) {
            if (!Schema::hasColumn('orders','stay_id')) {
                $table->foreignId('stay_id')->nullable()->after('user_id')->constrained('stays')->nullOnDelete();
            }
            if (!Schema::hasColumn('orders','delivery_mode')) {
                $table->enum('delivery_mode', ['room','external'])->default('external')->after('delivery_fee');
            }
            if (!Schema::hasColumn('orders','ordered_by_guest')) {
                $table->boolean('ordered_by_guest')->default(false)->after('delivery_mode');
            }
            if (!Schema::hasColumn('orders','guest_room_number')) {
                $table->string('guest_room_number', 50)->nullable()->after('ordered_by_guest');
            }
            if (!Schema::hasColumn('orders','delivery_address_verified')) {
                $table->boolean('delivery_address_verified')->default(false)->after('delivery_address');
            }
            if (!Schema::hasColumn('orders','delivery_latitude')) {
                $table->decimal('delivery_latitude', 10, 7)->nullable()->after('delivery_address_verified');
            }
            if (!Schema::hasColumn('orders','delivery_longitude')) {
                $table->decimal('delivery_longitude', 10, 7)->nullable()->after('delivery_latitude');
            }
            if (!Schema::hasColumn('orders','delivery_distance_km')) {
                $table->decimal('delivery_distance_km', 6, 2)->nullable()->after('delivery_longitude');
            }
            if (!Schema::hasColumn('orders','verification_log')) {
                $table->json('verification_log')->nullable()->after('delivery_distance_km');
            }
            if (!Schema::hasColumn('orders','recipient_name')) {
                $table->string('recipient_name', 120)->nullable()->after('verification_log');
            }
            if (!Schema::hasColumn('orders','recipient_phone')) {
                $table->string('recipient_phone', 40)->nullable()->after('recipient_name');
            }
            if (!Schema::hasColumn('orders','state')) {
                $table->enum('state', ['draft','pending_verification','awaiting_review','awaiting_payment','awaiting_kitchen','preparing','ready','delivering','completed','cancelled'])
                      ->default('awaiting_kitchen')->after('status');
            }
        });
    }

    public function down(): void
    {
        if (Schema::hasTable('orders')) {
            Schema::table('orders', function (Blueprint $table) {
                foreach ([
                    'stay_id','delivery_mode','ordered_by_guest','guest_room_number','delivery_address_verified',
                    'delivery_latitude','delivery_longitude','delivery_distance_km','verification_log',
                    'recipient_name','recipient_phone','state'
                ] as $col) {
                    if (Schema::hasColumn('orders', $col)) {
                        $table->dropColumn($col);
                    }
                }
            });
        }
        Schema::dropIfExists('stays');
    }
};
