<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('inventory_stats', function (Blueprint $table) {
            $table->id();
            $table->foreignId('dish_id')->constrained()->onDelete('cascade');
            $table->foreignId('location_id')->nullable()->constrained('locations')->onDelete('cascade');
            
            // Core inventory metrics
            $table->integer('on_hand')->default(0)->comment('Current physical stock');
            $table->integer('allocated')->default(0)->comment('Reserved for pending orders');
            $table->integer('available')->default(0)->comment('Available for new orders (on_hand - allocated)');
            
            // Movement aggregates
            $table->integer('total_sales')->default(0)->comment('Lifetime sales quantity');
            $table->integer('total_adjustments')->default(0)->comment('Lifetime manual adjustments');
            $table->integer('total_movements')->default(0)->comment('Total movement count');
            
            // Velocity metrics (cached from dishes table)
            $table->decimal('velocity_7d', 8, 2)->default(0)->comment('7-day sales velocity (units/day)');
            $table->decimal('velocity_30d', 8, 2)->default(0)->comment('30-day sales velocity (units/day)');
            $table->decimal('projected_runway_days', 8, 1)->nullable()->comment('Days until stockout');
            
            // Financial metrics
            $table->decimal('total_sales_value', 12, 2)->default(0)->comment('Lifetime sales revenue');
            $table->decimal('average_sale_price', 8, 2)->nullable()->comment('Average selling price');
            $table->decimal('inventory_value', 10, 2)->default(0)->comment('Current inventory value (cost basis)');
            
            // Performance tracking
            $table->integer('stockout_events')->default(0)->comment('Number of stockout occurrences');
            $table->integer('low_stock_events')->default(0)->comment('Number of low stock events');
            $table->integer('reorder_events')->default(0)->comment('Number of reorder point triggers');
            
            // Metadata
            $table->timestamp('last_movement_at')->nullable()->comment('Last inventory movement timestamp');
            $table->timestamp('last_sale_at')->nullable()->comment('Last sale timestamp');
            $table->timestamp('last_stockout_at')->nullable()->comment('Last stockout timestamp');
            $table->timestamp('stats_calculated_at')->nullable()->comment('When stats were last calculated');
            
            $table->timestamps();
            
            // Indexes for performance
            // Unique per dish globally when location is NULL, and unique per dish+location when location_id is set
            $table->unique(['dish_id', 'location_id']);
            $table->index('on_hand');
            $table->index('available');
            $table->index('velocity_7d');
            $table->index('projected_runway_days');
            $table->index('last_movement_at');
            $table->index('stats_calculated_at');
            
            // Composite indexes for common queries
            $table->index(['on_hand', 'available'], 'idx_stock_levels');
            $table->index(['velocity_7d', 'projected_runway_days'], 'idx_velocity_metrics');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('inventory_stats');
    }
};
