<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('inventory_snapshots', function (Blueprint $table) {
            $table->id();
            $table->date('snapshot_date')->comment('Date of the snapshot (YYYY-MM-DD)');
            $table->foreignId('dish_id')->nullable()->constrained()->onDelete('cascade');
            
            // Snapshot type (dish-specific or global aggregate)
            $table->enum('type', ['dish', 'global'])->default('dish')->comment('Snapshot granularity');
            
            // Stock levels at snapshot time
            $table->integer('on_hand')->default(0)->comment('Stock on hand at snapshot');
            $table->integer('allocated')->default(0)->comment('Allocated stock at snapshot');
            $table->integer('available')->default(0)->comment('Available stock at snapshot');
            
            // Daily activity metrics
            $table->integer('sales_qty')->default(0)->comment('Sales quantity for the day');
            $table->decimal('sales_value', 10, 2)->default(0)->comment('Sales value for the day');
            $table->integer('adjustments_qty')->default(0)->comment('Stock adjustments for the day');
            $table->integer('movements_count')->default(0)->comment('Total movements for the day');
            
            // Velocity metrics at snapshot time
            $table->decimal('velocity_7d', 8, 2)->default(0)->comment('7-day velocity at snapshot');
            $table->decimal('velocity_30d', 8, 2)->default(0)->comment('30-day velocity at snapshot');
            $table->decimal('projected_runway_days', 8, 1)->nullable()->comment('Projected runway at snapshot');
            
            // Financial metrics
            $table->decimal('inventory_value', 10, 2)->default(0)->comment('Inventory value at snapshot');
            $table->decimal('average_sale_price', 8, 2)->nullable()->comment('Average sale price');
            
            // Alert status at snapshot time
            $table->boolean('was_out_of_stock')->default(false)->comment('Was out of stock during day');
            $table->boolean('was_low_stock')->default(false)->comment('Was low stock during day');
            $table->boolean('had_stockout_alert')->default(false)->comment('Had active stockout alert');
            $table->integer('active_alerts_count')->default(0)->comment('Number of active alerts');
            
            // Performance metrics
            $table->decimal('fill_rate', 5, 2)->nullable()->comment('Order fill rate for the day (%)');
            $table->integer('stockout_events')->default(0)->comment('Stockout events during day');
            
            // Global aggregates (when type = 'global')
            $table->integer('total_dishes')->nullable()->comment('Total dishes tracked (global only)');
            $table->integer('dishes_with_stock')->nullable()->comment('Dishes with stock > 0 (global only)');
            $table->integer('dishes_low_stock')->nullable()->comment('Dishes below threshold (global only)');
            $table->integer('dishes_out_of_stock')->nullable()->comment('Dishes with 0 stock (global only)');
            $table->decimal('total_inventory_value', 12, 2)->nullable()->comment('Total inventory value (global only)');
            $table->decimal('avg_runway_days', 8, 1)->nullable()->comment('Average runway days (global only)');
            
            $table->timestamps();
            
            // Indexes for performance
            $table->unique(['snapshot_date', 'dish_id', 'type'], 'unique_snapshot_per_dish_date');
            $table->index('snapshot_date');
            $table->index('dish_id');
            $table->index('type');
            $table->index('on_hand');
            $table->index('sales_qty');
            $table->index('velocity_7d');
            $table->index('was_out_of_stock');
            
            // Composite indexes for common queries
            $table->index(['snapshot_date', 'type'], 'idx_date_type');
            $table->index(['dish_id', 'snapshot_date'], 'idx_dish_date');
            $table->index(['snapshot_date', 'sales_qty'], 'idx_date_sales');
            $table->index(['snapshot_date', 'was_out_of_stock', 'was_low_stock'], 'idx_date_alerts');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('inventory_snapshots');
    }
};
