<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up(): void
    {
        // Add location_id column if missing
        if (!Schema::hasColumn('inventory_stats', 'location_id')) {
            Schema::table('inventory_stats', function (Blueprint $table) {
                $table->foreignId('location_id')->nullable()->after('dish_id')->constrained('locations')->onDelete('cascade');
            });
        }

        // Drop old unique index on dish_id if it exists, then add composite unique
        try {
            DB::statement('ALTER TABLE `inventory_stats` DROP INDEX `inventory_stats_dish_id_unique`');
        } catch (\Throwable $e) {
            // ignore if index does not exist
        }

        // Add composite unique index (dish_id, location_id)
        Schema::table('inventory_stats', function (Blueprint $table) {
            // Guard against duplicate index creation
            try {
                DB::statement('CREATE UNIQUE INDEX `inventory_stats_dish_location_unique` ON `inventory_stats` (`dish_id`, `location_id`)');
            } catch (\Throwable $e) {
                // ignore if already exists
            }
        });
    }

    public function down(): void
    {
        // Drop composite unique index
        try {
            DB::statement('ALTER TABLE `inventory_stats` DROP INDEX `inventory_stats_dish_location_unique`');
        } catch (\Throwable $e) {
            // ignore
        }

        // Restore unique on dish_id (global only)
        try {
            DB::statement('CREATE UNIQUE INDEX `inventory_stats_dish_id_unique` ON `inventory_stats` (`dish_id`)');
        } catch (\Throwable $e) {
            // ignore
        }

        // Drop location_id column
        if (Schema::hasColumn('inventory_stats', 'location_id')) {
            Schema::table('inventory_stats', function (Blueprint $table) {
                $table->dropConstrainedForeignId('location_id');
            });
        }
    }
};
