# InventoryService Usage Guide

This guide shows how to use the centralized InventoryService to perform stock operations and record auditable movements.

- Namespace: App\Services\Inventory\InventoryService
- Model: App\Models\InventoryMovement
- Tables: dishes (stock, low_stock_threshold), inventory_movements

## Core methods

1) recordSale(Dish $dish, int $quantity, ?Order $order = null, ?User $user = null, ?string $note = null): InventoryMovement
- Decrements stock by $quantity
- Creates a movement with reason "sale"
- Links to optional order and user
- Throws ValidationException if $quantity <= 0 or insufficient stock

Example:

$movement = app(InventoryService::class)->recordSale(
    dish: $dish,
    quantity: 3,
    order: $order,
    user: auth()->user(),
    note: "Order #{$order->id} checkout"
);

2) adjust(Dish $dish, int $change, string $reason = 'adjustment', ?User $user = null, ?string $note = null): InventoryMovement
- Increases or decreases stock by $change
- Validates $reason against canonical list
- Enforces non-negative final stock

Example:

$movement = app(InventoryService::class)->adjust(
    dish: $dish,
    change: -2,
    reason: 'spoilage',
    user: auth()->user(),
    note: 'Two spoiled items removed'
);

3) getAvailableStock(Dish $dish): int
- Returns current on-hand stock (Phase 1 = dish.stock)

4) isAvailable(Dish $dish, int $requestedQuantity): bool
- Returns true if stock is sufficient

## Canonical reasons

- sale, adjustment, correction, receipt, spoilage, shrinkage, return

Check validity:

$svc = app(InventoryService::class);
$svc->isValidReason('sale'); // true
$svc->getValidReasons(); // array label map

## Transactions and locking

All stock updates run inside a DB::transaction() and acquire a pessimistic row lock on the target dish (lockForUpdate). This prevents oversell race conditions and guarantees a consistent non-negative stock rule at commit time.

## Error handling

Methods throw Illuminate\Validation\ValidationException with message keys:
- 'quantity' => 'Sale quantity must be positive'
- 'stock' => 'Insufficient stock. Available: X, Requested: Y'
- 'change' => 'Stock change cannot be zero'
- 'reason' => 'Invalid movement reason: ...'

Wrap calls in try/catch when used in controllers to present a friendly message and rollback the transaction.

## Tests (already implemented)

- Unit: Tests\Unit\Services\Inventory\InventoryServiceTest covers success, error, and concurrency cases.
- Feature: Tests\Feature\Inventory\CheckoutInventoryIntegrationTest verifies checkout uses InventoryService and records movements.

## Admin UI integration

- Inventory Dashboard: KPIs, recent movements, low stock alerts
- Movements list: filters, pagination, CSV export
- Dishes list: stock badges; adjustment modal posts to admin.dishes.inventory.adjust

## Next phases

- Allocation, BOM/recipes, snapshots, and forecasting are deferred and will extend this service.
