<?php

/**
 * Manual Inventory Testing Script
 * 
 * Run this with: php artisan tinker
 * Then copy and paste these commands to test the inventory system manually
 */

echo "=== Manual Inventory System Testing ===\n";

// 1. Test InventoryService directly
use App\Services\Inventory\InventoryService;
use App\Models\Dish;
use App\Models\User;
use App\Models\Order;

$inventoryService = app(InventoryService::class);

// Create test data
$dish = Dish::factory()->create(['name' => 'Test Dish', 'stock' => 10, 'price' => 100]);
$user = User::factory()->create(['name' => 'Test User']);

echo "Created test dish with ID {$dish->id} and initial stock: {$dish->stock}\n";

// Test 1: Check available stock
$available = $inventoryService->getAvailableStock($dish->id);
echo "Available stock: {$available}\n";

// Test 2: Check if quantity is available
$isAvailable = $inventoryService->isAvailable($dish->id, 5);
echo "Is 5 units available? " . ($isAvailable ? 'Yes' : 'No') . "\n";

// Test 3: Record a sale
$order = Order::factory()->create(['user_id' => $user->id]);
try {
    $inventoryService->recordSale($dish->id, 3, $order->id, $user->id);
    echo "Successfully recorded sale of 3 units\n";
    
    $dish->refresh();
    echo "Stock after sale: {$dish->stock}\n";
} catch (Exception $e) {
    echo "Sale failed: " . $e->getMessage() . "\n";
}

// Test 4: Manual adjustment
try {
    $inventoryService->adjust($dish->id, 2, 'adjustment', $user->id, 'Manual stock increase');
    echo "Successfully adjusted stock by +2\n";
    
    $dish->refresh();
    echo "Stock after adjustment: {$dish->stock}\n";
} catch (Exception $e) {
    echo "Adjustment failed: " . $e->getMessage() . "\n";
}

// Test 5: Check movement history
$movements = \App\Models\InventoryMovement::where('dish_id', $dish->id)->get();
echo "Movement history for dish {$dish->id}:\n";
foreach ($movements as $movement) {
    echo "- {$movement->change} units, reason: {$movement->reason}, note: {$movement->note}\n";
}

// Test 6: Try insufficient stock
try {
    $inventoryService->recordSale($dish->id, 100, $order->id, $user->id);
    echo "ERROR: Should not allow overselling!\n";
} catch (Exception $e) {
    echo "Correctly prevented overselling: " . $e->getMessage() . "\n";
}

echo "\n=== Manual Testing Complete ===\n";