<?php
/**
 * Professional QR Code Generator following global best practices
 * Generates real QR codes for PWA installation
 */

// Security headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');

// Get parameters with validation
$size = isset($_GET['size']) ? max(100, min(1000, (int)$_GET['size'])) : 300;
$type = isset($_GET['type']) ? preg_replace('/[^a-z]/', '', $_GET['type']) : 'pwa';

# Production URL for Bonny Delight
# Use subdomain for production, localhost for development
$isProduction = (isset($_SERVER['HTTP_HOST']) && ($_SERVER['HTTP_HOST'] === 'app.bonnydelight.ng' || $_SERVER['HTTP_HOST'] === 'www.app.bonnydelight.ng'));
$baseUrl = $isProduction ? 'https://app.bonnydelight.ng' : 'http://localhost:9090/bonny_delight/public';

// Generate QR data based on type following PWA best practices
switch ($type) {
    case 'pwa':
        // Best practice: Use the start_url from manifest.json with PWA source tracking
        $qrData = $baseUrl . '/?source=pwa_qr';
        $title = 'Install Bonny Delight App';
        break;
    case 'contact':
        $qrData = $baseUrl . '/contact';
        $title = 'Contact Bonny Delight';
        break;
    case 'menu':
        $qrData = $baseUrl . '/menu';
        $title = 'View Menu';
        break;
    default:
        $qrData = $baseUrl;
        $title = 'Bonny Delight';
        break;
}

// QR Code configuration following global standards
$errorCorrection = 'M'; // Medium error correction (15%) - good balance for logo overlay
$margin = 2; // Standard margin
$qrSize = $size - 40; // Leave space for border and logo
$logoSize = (int)($qrSize * 0.2); // Logo should be max 20% of QR size (best practice)

// Create QR code URL using a reliable service with proper parameters
$qrApiUrl = 'https://api.qrserver.com/v1/create-qr-code/?' . http_build_query([
    'size' => $qrSize . 'x' . $qrSize,
    'data' => $qrData,
    'format' => 'svg',
    'ecc' => $errorCorrection,
    'margin' => $margin,
    'bgcolor' => 'ffffff',
    'color' => '1f2937',
    'qzone' => '1'
]);

// Set proper content type
header('Content-Type: image/svg+xml');
header('Cache-Control: public, max-age=86400'); // Cache for 24 hours
header('X-QR-Data: ' . $qrData); // Debug header

// Logo configuration
$logoCenter = $logoSize / 2;
$logoRadius = $logoSize / 2;
$logoPadding = 8;
$logoBackgroundRadius = $logoRadius + $logoPadding;

// Fetch QR code with error handling
$qrSvgContent = '';
$context = stream_context_create([
    'http' => [
        'timeout' => 10,
        'user_agent' => 'BonnyDelight-QRGenerator/1.0'
    ]
]);

$qrSvgRaw = @file_get_contents($qrApiUrl, false, $context);

if ($qrSvgRaw) {
    // Clean and prepare the QR SVG for embedding
    $qrSvgContent = $qrSvgRaw;
    $qrSvgContent = preg_replace('/<\?xml[^>]*\?>/', '', $qrSvgContent);
    $qrSvgContent = preg_replace('/<!DOCTYPE[^>]*>/', '', $qrSvgContent);
    $qrSvgContent = str_replace('<svg', '<g transform="translate(20, 20)"><svg', $qrSvgContent);
    $qrSvgContent = str_replace('</svg>', '</svg></g>', $qrSvgContent);
} else {
    // Fallback: Create a basic QR-like pattern if external service fails
    $fallbackPattern = '';
    $gridSize = 25;
    $cellSize = $qrSize / $gridSize;
    
    for ($row = 0; $row < $gridSize; $row++) {
        for ($col = 0; $col < $gridSize; $col++) {
            $hash = crc32($qrData . $row . $col);
            if ($hash % 3 === 0) {
                $x = 20 + ($col * $cellSize);
                $y = 20 + ($row * $cellSize);
                $fallbackPattern .= "<rect x=\"{$x}\" y=\"{$y}\" width=\"{$cellSize}\" height=\"{$cellSize}\" fill=\"#1f2937\"/>";
            }
        }
    }
    
    $qrSvgContent = "<g><rect x=\"20\" y=\"20\" width=\"{$qrSize}\" height=\"{$qrSize}\" fill=\"white\" rx=\"8\"/>{$fallbackPattern}</g>";
}

// Load the colorful logo
$logoSvgPath = __DIR__ . '/assets/logo-colorful.svg';
$logoSvg = '';
if (file_exists($logoSvgPath)) {
    $logoSvgRaw = file_get_contents($logoSvgPath);
    // Extract the logo content and scale it
    $logoSvg = preg_replace('/<svg[^>]*>/', '', $logoSvgRaw);
    $logoSvg = str_replace('</svg>', '', $logoSvg);
    $logoScale = $logoSize / 200; // Assuming original logo is 200x200
}

# Calculate logo position (center of QR code)
$logoX = ($size - $logoSize) / 2;
$logoY = ($size - $logoSize) / 2;

# Calculate dimensions for SVG
$borderX = 4;
$borderY = 4;
$borderWidth = $size - 8;
$borderHeight = $size - 8;
$logoTranslateX = 100 - (100 * $logoScale);
$logoTranslateY = 100 - (100 * $logoScale);

// Generate the final SVG with modern styling
$svg = <<<SVG
<svg width="{$size}" height="{$size}" viewBox="0 0 {$size} {$size}" xmlns="http://www.w3.org/2000/svg">
  <defs>
    <linearGradient id="bgGradient" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:#ffffff;stop-opacity:1" />
      <stop offset="100%" style="stop-color:#f8fafc;stop-opacity:1" />
    </linearGradient>
    <linearGradient id="borderGradient" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:#10b981;stop-opacity:1" />
      <stop offset="100%" style="stop-color:#059669;stop-opacity:1" />
    </linearGradient>
    <filter id="shadow" x="-50%" y="-50%" width="200%" height="200%">
      <feDropShadow dx="0" dy="4" stdDeviation="8" flood-color="#000000" flood-opacity="0.15"/>
    </filter>
    <filter id="logoShadow" x="-50%" y="-50%" width="200%" height="200%">
      <feDropShadow dx="0" dy="2" stdDeviation="4" flood-color="#000000" flood-opacity="0.3"/>
    </filter>
  </defs>
  
  <!-- Background with gradient -->
  <rect width="{$size}" height="{$size}" fill="url(#bgGradient)" rx="20" ry="20" filter="url(#shadow)"/>
  
  <!-- Border -->
  <rect x="{$borderX}" y="{$borderY}" width="{$borderWidth}" height="{$borderHeight}" fill="none" stroke="url(#borderGradient)" stroke-width="2" rx="16" ry="16"/>
  
  <!-- QR Code -->
  {$qrSvgContent}
  
  <!-- Logo overlay with background -->
  <g transform="translate({$logoX}, {$logoY})">
    <!-- White background circle for logo -->
    <circle cx="{$logoCenter}" cy="{$logoCenter}" r="{$logoBackgroundRadius}" fill="white" filter="url(#logoShadow)"/>
    <!-- Logo content -->
    <g transform="scale({$logoScale}) translate({$logoTranslateX}, {$logoTranslateY})">
      {$logoSvg}
    </g>
  </g>
  
  <!-- QR Code title (optional, for accessibility) -->
  <title>{$title}</title>
</svg>
SVG;

echo $svg;
?>