@props(['order' => null])
@if($order)
<section class="surface rounded-xl p-6" data-io-hydrate @io-hydrate.window="$dispatch('active-start')" x-data="{
  order:@js($order), now: Date.now(), timer:null, poller:null, err:null,
  announce:'', lastStatus:null, terminals:['Delivered','Cancelled'],
  // i18n strings
  i18n: {
    updatedTpl: @js('Order status updated to :status'),
    unavailable: @js('Live updates are temporarily unavailable.'),
  },
  // polling/backoff
  pollBase: 25000, backoff: 0, maxBackoff: 3, pollTimeout: null,
  setStatus(s){
    if(typeof s !== 'string') return;
    if(this.lastStatus===null) this.lastStatus = this.order?.status || s;
    if(s !== this.lastStatus){
      const tpl = this.i18n.updatedTpl || 'Order status updated to :status';
      this.announce = tpl.replace(':status', s);
      this.lastStatus = s;
    }
    if(this.terminals.includes(s)){
      if(this.poller){ clearTimeout(this.poller); this.poller=null; }
      if(this.pollTimeout){ clearTimeout(this.pollTimeout); this.pollTimeout=null; }
    }
  },
  async poll(){
    if(!this.order || !this.order.id || !this.order.status_url) return;
    if (document.hidden || (typeof navigator.onLine === 'boolean' && !navigator.onLine)) { this.scheduleNext(); return; }
    try{
      const res = await fetch(this.order.status_url, {headers:{'Accept':'application/json'}});
      const data = await res.json();
      if(data && data.ok){
        this.order.status = data.status;
        this.order.eta = data.eta;
        this.err = null;
        this.setStatus(data.status || '');
        this.backoff = 0; // reset backoff on success
      } else if (data && data.db_unavailable){
        this.err = this.i18n.unavailable;
        this.backoff = Math.min(this.backoff + 1, this.maxBackoff);
      }
    }catch(e){ this.err = this.i18n.unavailable; this.backoff = Math.min(this.backoff + 1, this.maxBackoff); }
    this.scheduleNext();
  },
  scheduleNext(){
    if(this.terminals.includes(this.order?.status)) return;
    const delay = this.pollBase + (this.backoff * 10000); // 25s, 35s, 45s, 55s
    if (this.pollTimeout) { clearTimeout(this.pollTimeout); }
    this.pollTimeout = setTimeout(() => { if(!document.hidden){ this.poll(); } else { this.scheduleNext(); } }, delay);
  }
}" x-init="
  timer=setInterval(()=>now=Date.now(),1000);
  // only start polling once visible (IntersectionObserver will dispatch 'io-hydrate')
  if (window.__noIO) { scheduleNext(); }
  document.addEventListener('visibilitychange', ()=>{ if(!document.hidden){ $data.backoff = 0; $data.poll(); } });
" @active-start.window="
  scheduleNext();
" @destroy.window="
  clearInterval(timer);
  if(poller) clearTimeout(poller);
  if(pollTimeout) clearTimeout(pollTimeout);
">
  <div class="flex items-center justify-between gap-4">
    <div class="flex items-center gap-4">
      <div class="w-12 h-12 rounded-full bg-emerald-500/10 text-emerald-600 flex items-center justify-center">
        <svg class="w-6 h-6" viewBox="0 0 24 24" fill="none" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12A9 9 0 1 1 3 12a9 9 0 0 1 18 0z"/></svg>
      </div>
      <div>
        <h3 class="font-semibold text-slate-900 dark:text-white">Active Order #<span x-text="order.id"></span></h3>
        <p class="text-sm text-slate-600 dark:text-slate-400" aria-live="polite" role="status">
          <span>Status:</span>
          <span class="font-medium" x-text="order.status"></span>
          <template x-if="order.eta">
            <span> • ETA <span x-text="new Date(order.eta).toLocaleTimeString([], {hour:'2-digit', minute:'2-digit'})"></span></span>
          </template>
        </p>
        <div class="sr-only" aria-live="polite" x-text="announce"></div>
        <template x-if="err">
          <p class="mt-1 text-xs text-amber-600 dark:text-amber-300" x-text="i18n.unavailable"></p>
        </template>
      </div>
    </div>
    <div class="flex items-center gap-2">
      <a :href="order.track_url || '{{ route('orders.index') }}'" class="btn btn-outline">Track</a>
    </div>
  </div>
</section>
@endif
