@extends('super-admin.layout')

@section('title', 'Business Analytics')
@section('subtitle', 'Comprehensive insights and performance metrics')

@section('content')
<div class="space-y-6" id="analyticsApp">
    <div class="bg-white dark:bg-slate-900 rounded-lg border border-slate-200 dark:border-slate-700 p-6">
        <div class="flex flex-col lg:flex-row items-start lg:items-center justify-between gap-4">
            <div>
                <h1 class="text-2xl font-bold">Analytics Dashboard</h1>
                <p class="text-sm text-slate-600 dark:text-slate-400 mt-1">Key KPIs and trends</p>
            </div>
            <div class="flex flex-wrap items-center gap-3">
                <div class="flex bg-slate-100 dark:bg-slate-800 rounded-lg p-1">
                    <button type="button" data-period="7d" class="period-btn px-3 py-2 text-sm rounded-md hover:bg-white dark:hover:bg-slate-700">7 Days</button>
                    <button type="button" data-period="30d" class="period-btn px-3 py-2 text-sm rounded-md bg-indigo-600 text-white">30 Days</button>
                    <button type="button" data-period="90d" class="period-btn px-3 py-2 text-sm rounded-md hover:bg-white dark:hover:bg-slate-700">90 Days</button>
                    <button type="button" data-period="1y" class="period-btn px-3 py-2 text-sm rounded-md hover:bg-white dark:hover:bg-slate-700">1 Year</button>
                </div>
                <label class="flex items-center gap-2 text-sm">
                    <input type="checkbox" id="autoRefresh" class="rounded border-slate-300 dark:border-slate-600" checked>
                    <span>Auto-refresh</span>
                </label>
                <div class="text-xs text-slate-500 dark:text-slate-400">Last updated: <span id="lastUpdated" class="font-medium">—</span></div>
            </div>
        </div>
    </div>

    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6">
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">Today's Revenue</p>
            <p class="text-2xl font-bold" id="todayRevenue">₦0</p>
            <p class="text-sm"><span id="revenueGrowth" class="text-green-600 dark:text-green-400">+0%</span> <span class="text-slate-500">vs yesterday</span></p>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">Today's Orders</p>
            <p class="text-2xl font-bold" id="todayOrders">0</p>
            <p class="text-sm text-slate-500">Pending: <span id="pendingOrders" class="text-orange-600">0</span></p>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">Avg Order Value</p>
            <p class="text-2xl font-bold" id="avgOrderValue">₦0</p>
            <p class="text-sm text-slate-500">This period</p>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">Customer Retention</p>
            <p class="text-2xl font-bold" id="retentionRate">0%</p>
            <p class="text-sm text-slate-500">Repeat customers</p>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">New Customers</p>
            <p class="text-2xl font-bold" id="newCustomers">0</p>
            <p class="text-sm text-slate-500">Today</p>
        </div>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Revenue Trends</h2>
            <div class="h-64"><canvas id="revenueChart"></canvas></div>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Revenue by Category</h2>
            <div class="h-64"><canvas id="categoryChart"></canvas></div>
        </div>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Top Customers</h2>
            <div id="topCustomers" class="space-y-2">
                <div class="h-4 bg-slate-200 dark:bg-slate-700 rounded w-3/4 animate-pulse"></div>
                <div class="h-4 bg-slate-200 dark:bg-slate-700 rounded w-1/2 animate-pulse"></div>
            </div>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Best Sellers</h2>
            <div class="h-64"><canvas id="productsChart"></canvas></div>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Recent Activity</h2>
            <div id="recentActivity" class="space-y-2">
                <div class="h-4 bg-slate-200 dark:bg-slate-700 rounded animate-pulse"></div>
                <div class="h-4 bg-slate-200 dark:bg-slate-700 rounded w-3/4 animate-pulse"></div>
            </div>
        </div>
    </div>

    <!-- Top Selling Dishes - Detailed View -->
    <div class="bg-white dark:bg-slate-900 rounded-lg border border-slate-200 dark:border-slate-700 p-6">
        <div class="flex items-center justify-between mb-6">
            <div>
                <h2 class="text-xl font-bold">Top Selling Dishes</h2>
                <p class="text-sm text-slate-600 dark:text-slate-400 mt-1">Detailed performance metrics for your best-performing menu items</p>
            </div>
            <button type="button" id="exportDishes" class="inline-flex items-center gap-2 px-4 py-2 text-sm font-medium text-slate-700 dark:text-slate-300 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 rounded-lg transition">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                Export CSV
            </button>
        </div>

        <!-- Table View -->
        <div class="overflow-x-auto">
            <table class="w-full text-sm" id="dishesTable">
                <thead class="bg-slate-50 dark:bg-slate-800 border-y border-slate-200 dark:border-slate-700">
                    <tr>
                        <th class="px-4 py-3 text-left font-semibold">#</th>
                        <th class="px-4 py-3 text-left font-semibold">Dish Name</th>
                        <th class="px-4 py-3 text-left font-semibold">Category</th>
                        <th class="px-4 py-3 text-right font-semibold">Units Sold</th>
                        <th class="px-4 py-3 text-right font-semibold">Revenue</th>
                        <th class="px-4 py-3 text-right font-semibold">Avg Price</th>
                        <th class="px-4 py-3 text-right font-semibold">Orders</th>
                        <th class="px-4 py-3 text-right font-semibold">Avg Qty/Order</th>
                        <th class="px-4 py-3 text-center font-semibold">Trend</th>
                    </tr>
                </thead>
                <tbody id="dishesTableBody" class="divide-y divide-slate-200 dark:divide-slate-700">
                    <!-- Loading state -->
                    <tr>
                        <td colspan="9" class="px-4 py-8 text-center text-slate-500">
                            <div class="flex items-center justify-center gap-2">
                                <svg class="animate-spin h-5 w-5" fill="none" viewBox="0 0 24 24">
                                    <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                                    <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                </svg>
                                Loading top dishes...
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>

        <!-- Pagination & Summary -->
        <div class="flex flex-col sm:flex-row items-center justify-between gap-4 mt-6 pt-6 border-t border-slate-200 dark:border-slate-700">
            <div class="text-sm text-slate-600 dark:text-slate-400">
                Showing <span id="dishesShowing" class="font-medium">0</span> of <span id="dishesTotal" class="font-medium">0</span> dishes
            </div>
            <div class="flex items-center gap-2">
                <button type="button" id="dishesShowAll" class="px-3 py-1.5 text-sm font-medium text-indigo-600 dark:text-indigo-400 hover:bg-indigo-50 dark:hover:bg-indigo-900/20 rounded transition">
                    Show All
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
(function(){
    'use strict';
    const base = '{{ url('/super-admin/analytics') }}';
    const fmt = (n) => Number(n||0).toLocaleString('en-NG', { minimumFractionDigits:2, maximumFractionDigits:2 });
    const qsAll = (sel) => Array.from(document.querySelectorAll(sel));
    const state = { period: '30d', charts: {}, timer: null, allDishes: [], showAllDishes: false };

    function setActive(btn){
        qsAll('.period-btn').forEach(b=>{ b.classList.remove('bg-indigo-600','text-white'); });
        btn.classList.add('bg-indigo-600','text-white');
    }
    function text(id, val){ const el = document.getElementById(id); if(el) el.textContent = val; }
    function now(){ const el=document.getElementById('lastUpdated'); if(el) el.textContent = new Date().toLocaleTimeString(); }
    function destroy(name){ if(state.charts[name]){ state.charts[name].destroy(); delete state.charts[name]; } }
    async function getJson(url){ const r = await fetch(url, { headers:{ 'X-Requested-With':'XMLHttpRequest' } }); if(!r.ok) throw new Error('HTTP '+r.status); return r.json(); }

    async function loadRealtime(){
        try{ const d = await getJson(`${base}/realtime`);
            text('todayRevenue', '₦'+fmt(d.today_metrics?.revenue_today||0));
            text('todayOrders', d.today_metrics?.orders_today||0);
            text('pendingOrders', d.today_metrics?.pending_orders||0);
            text('avgOrderValue', '₦'+fmt(d.today_metrics?.avg_order_value_today||0));
            text('newCustomers', d.today_metrics?.new_customers_today||0);
            if(Array.isArray(d.recent_orders)){
                const box = document.getElementById('recentActivity');
                if(box) box.innerHTML = d.recent_orders.slice(0,5).map(o=>`<div class="flex items-center justify-between text-sm py-2 border-b border-slate-200 dark:border-slate-700 last:border-0"><span>#${o.id} - ${escapeHtml(o.user?.name||'Guest')}</span><span class="font-medium">₦${fmt(o.total||0)}</span></div>`).join('');
            }
        }catch(e){ console.error('Realtime failed', e); }
    }
    async function loadRevenue(){
        try{ const d = await getJson(`${base}/revenue?period=${encodeURIComponent(state.period)}`);
            const g = Number(d.revenue_growth||0); const el = document.getElementById('revenueGrowth');
            if(el){ el.textContent = (g>=0?'+':'')+g.toFixed(1)+'%'; el.className = (g>=0? 'text-green-600 dark:text-green-400':'text-red-600 dark:text-red-400')+' font-medium'; }
            if(Array.isArray(d.revenue_data)) renderRevenue(d.revenue_data);
            if(Array.isArray(d.category_revenue)) renderCategory(d.category_revenue);
        }catch(e){ console.error('Revenue failed', e); }
    }
    async function loadCustomers(){
        try{ const d = await getJson(`${base}/customers?period=${encodeURIComponent(state.period)}`);
            text('retentionRate', (Number(d.repeat_rate||0)).toFixed(1)+'%');
            if(Array.isArray(d.customer_ltv)){
                const box=document.getElementById('topCustomers');
                if(box) box.innerHTML = d.customer_ltv.slice(0,5).map(c=>`<div class="flex items-center justify-between py-2 border-b border-slate-200 dark:border-slate-700 last:border-0"><div><div class="font-medium">${escapeHtml(c.name||'—')}</div><div class="text-sm text-slate-500">${Number(c.total_orders||0)} orders</div></div><div class="font-medium">₦${fmt(c.lifetime_value||0)}</div></div>`).join('');
            }
        }catch(e){ console.error('Customers failed', e); }
    }
    async function loadProducts(){
        try{ 
            const d = await getJson(`${base}/products?period=${encodeURIComponent(state.period)}`);
            if(Array.isArray(d.best_sellers)) {
                renderProducts(d.best_sellers);
                renderDishesTable(d.best_sellers);
                state.allDishes = d.best_sellers; // Store for export
            }
        }catch(e){ console.error('Products failed', e); }
    }

    function renderRevenue(rows){ destroy('revenue'); const ctx=document.getElementById('revenueChart'); if(!ctx||!window.Chart) return; state.charts.revenue = new Chart(ctx,{ type:'line', data:{ labels: rows.map(r=> new Date(r.date).toLocaleDateString()), datasets:[{ label:'Revenue (₦)', data: rows.map(r=> Number(r.revenue)||0), borderColor:'#3B82F6', backgroundColor:'rgba(59,130,246,0.1)', fill:true, tension:0.35 }] }, options:{ responsive:true, maintainAspectRatio:false, scales:{ y:{ beginAtZero:true, ticks:{ callback:v=>'₦'+fmt(v) } } } } }); }
    function renderCategory(rows){ destroy('category'); const ctx=document.getElementById('categoryChart'); if(!ctx||!window.Chart) return; state.charts.category = new Chart(ctx,{ type:'doughnut', data:{ labels: rows.map(r=> r.name), datasets:[{ data: rows.map(r=> Number(r.revenue)||0), backgroundColor:['#3B82F6','#10B981','#F59E0B','#EF4444','#8B5CF6','#06B6D4','#84CC16','#F97316','#EC4899','#6366F1'] }] }, options:{ responsive:true, maintainAspectRatio:false } }); }
    function renderProducts(rows){ destroy('products'); const ctx=document.getElementById('productsChart'); if(!ctx||!window.Chart) return; const top=rows.slice(0,5); state.charts.products = new Chart(ctx,{ type:'bar', data:{ labels: top.map(r=> r.name.length>14 ? r.name.slice(0,14)+'…' : r.name), datasets:[{ label:'Units', data: top.map(r=> Number(r.total_sold)||0), backgroundColor:'rgba(16,185,129,0.8)' }] }, options:{ responsive:true, maintainAspectRatio:false, plugins:{ legend:{ display:false } }, scales:{ y:{ beginAtZero:true } } } }); }
    
    function renderDishesTable(dishes){
        const tbody = document.getElementById('dishesTableBody');
        const showing = document.getElementById('dishesShowing');
        const total = document.getElementById('dishesTotal');
        
        if(!tbody) return;
        
        // Display logic: show top 10 initially
        const displayLimit = state.showAllDishes ? dishes.length : Math.min(10, dishes.length);
        const displayDishes = dishes.slice(0, displayLimit);
        
        if(displayDishes.length === 0){
            tbody.innerHTML = '<tr><td colspan="9" class="px-4 py-8 text-center text-slate-500">No dishes data available for this period</td></tr>';
            if(showing) showing.textContent = '0';
            if(total) total.textContent = '0';
            return;
        }
        
        tbody.innerHTML = displayDishes.map((dish, idx) => {
            const rank = idx + 1;
            const unitsSold = Number(dish.total_sold)||0;
            const revenue = Number(dish.revenue)||0;
            const avgPrice = Number(dish.avg_selling_price)||0;
            const ordersCount = Number(dish.orders_count)||0;
            const avgQtyPerOrder = Number(dish.avg_quantity_per_order)||0;
            
            // Calculate trend indicator (simple: top 3 get fire emoji, 4-7 get up arrow, rest neutral)
            let trendIcon = '';
            if(rank <= 3){
                trendIcon = '<span class="text-lg" title="Hot seller!">🔥</span>';
            } else if(rank <= 7){
                trendIcon = '<span class="text-green-600 dark:text-green-400" title="Rising">↗</span>';
            } else {
                trendIcon = '<span class="text-slate-400" title="Steady">→</span>';
            }
            
            // Badge colors by rank
            let rankBadge = '';
            if(rank === 1){
                rankBadge = '<span class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-yellow-100 dark:bg-yellow-900/30 text-yellow-800 dark:text-yellow-200 font-bold text-sm">1</span>';
            } else if(rank === 2){
                rankBadge = '<span class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-slate-100 dark:bg-slate-800 text-slate-700 dark:text-slate-300 font-bold text-sm">2</span>';
            } else if(rank === 3){
                rankBadge = '<span class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-orange-100 dark:bg-orange-900/30 text-orange-800 dark:text-orange-200 font-bold text-sm">3</span>';
            } else {
                rankBadge = `<span class="inline-flex items-center justify-center w-8 h-8 rounded-full bg-slate-50 dark:bg-slate-800/50 text-slate-600 dark:text-slate-400 font-medium text-sm">${rank}</span>`;
            }
            
            return `
                <tr class="hover:bg-slate-50 dark:hover:bg-slate-800/50 transition">
                    <td class="px-4 py-3">${rankBadge}</td>
                    <td class="px-4 py-3">
                        <div class="font-medium text-slate-900 dark:text-slate-100">${escapeHtml(dish.name)}</div>
                        <div class="text-xs text-slate-500 dark:text-slate-400 mt-0.5">₦${fmt(dish.price)} base price</div>
                    </td>
                    <td class="px-4 py-3">
                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-indigo-100 dark:bg-indigo-900/30 text-indigo-800 dark:text-indigo-200">
                            ${escapeHtml(dish.category)}
                        </span>
                    </td>
                    <td class="px-4 py-3 text-right">
                        <div class="font-semibold text-slate-900 dark:text-slate-100">${unitsSold.toLocaleString()}</div>
                        <div class="text-xs text-slate-500">units</div>
                    </td>
                    <td class="px-4 py-3 text-right">
                        <div class="font-semibold text-green-600 dark:text-green-400">₦${fmt(revenue)}</div>
                        <div class="text-xs text-slate-500">${((revenue/dishes.reduce((sum,d)=>sum+Number(d.revenue||0),0))*100).toFixed(1)}% of total</div>
                    </td>
                    <td class="px-4 py-3 text-right">
                        <div class="font-medium">₦${fmt(avgPrice)}</div>
                    </td>
                    <td class="px-4 py-3 text-right">
                        <div class="font-medium">${ordersCount.toLocaleString()}</div>
                        <div class="text-xs text-slate-500">orders</div>
                    </td>
                    <td class="px-4 py-3 text-right">
                        <div class="font-medium">${avgQtyPerOrder.toFixed(1)}</div>
                        <div class="text-xs text-slate-500">qty/order</div>
                    </td>
                    <td class="px-4 py-3 text-center">${trendIcon}</td>
                </tr>
            `;
        }).join('');
        
        if(showing) showing.textContent = displayLimit.toString();
        if(total) total.textContent = dishes.length.toString();
        
        // Update show all button
        const showAllBtn = document.getElementById('dishesShowAll');
        if(showAllBtn){
            if(dishes.length <= 10){
                showAllBtn.style.display = 'none';
            } else {
                showAllBtn.style.display = 'block';
                showAllBtn.textContent = state.showAllDishes ? 'Show Less' : `Show All (${dishes.length})`;
            }
        }
    }
    
    function exportDishesCSV(){
        const dishes = state.allDishes || [];
        if(dishes.length === 0){
            alert('No data to export');
            return;
        }
        
        // CSV headers
        let csv = 'Rank,Dish Name,Category,Units Sold,Revenue (₦),Avg Price (₦),Orders Count,Avg Qty per Order\n';
        
        // CSV rows
        dishes.forEach((dish, idx) => {
            csv += [
                idx + 1,
                `"${(dish.name||'').replace(/"/g, '""')}"`,
                `"${(dish.category||'').replace(/"/g, '""')}"`,
                Number(dish.total_sold)||0,
                Number(dish.revenue)||0,
                Number(dish.avg_selling_price)||0,
                Number(dish.orders_count)||0,
                (Number(dish.avg_quantity_per_order)||0).toFixed(2)
            ].join(',') + '\n';
        });
        
        // Create download
        const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        link.setAttribute('href', url);
        link.setAttribute('download', `top_dishes_${state.period}_${new Date().toISOString().split('T')[0]}.csv`);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }
    
    function escapeHtml(t){ const d=document.createElement('div'); d.textContent=String(t??''); return d.innerHTML; }

    async function refreshAll(){ 
        state.showAllDishes = false; // Reset show all when period changes
        await Promise.all([loadRealtime(), loadRevenue(), loadCustomers(), loadProducts()]); 
        now(); 
    }
    
    function bind(){
        // Period buttons
        qsAll('.period-btn').forEach(btn=> btn.addEventListener('click',()=>{ 
            state.period = btn.getAttribute('data-period'); 
            setActive(btn); 
            refreshAll(); 
        }));
        
        // Auto-refresh toggle
        const ar=document.getElementById('autoRefresh');
        if(ar){ 
            const stop=()=>{ if(state.timer){ clearInterval(state.timer); state.timer=null; } }; 
            const start=()=>{ stop(); state.timer=setInterval(()=> loadRealtime().then(now).catch(()=>{}), 30000); }; 
            ar.addEventListener('change', e=> e.target.checked? start() : stop()); 
            start(); 
        }
        
        // Export dishes CSV
        const exportBtn = document.getElementById('exportDishes');
        if(exportBtn) exportBtn.addEventListener('click', exportDishesCSV);
        
        // Show all dishes toggle
        const showAllBtn = document.getElementById('dishesShowAll');
        if(showAllBtn) showAllBtn.addEventListener('click', ()=>{
            state.showAllDishes = !state.showAllDishes;
            if(state.allDishes && state.allDishes.length > 0){
                renderDishesTable(state.allDishes);
            }
        });
    }
    
    document.addEventListener('DOMContentLoaded', ()=>{ bind(); refreshAll(); });
})();
</script>
@endpush