@extends('super-admin.layout')

@section('title', 'Business Analytics')
@section('subtitle', 'Comprehensive insights and performance metrics')

@section('content')
<div class="space-y-6" id="analyticsApp">
    <div class="bg-white dark:bg-slate-900 rounded-lg border border-slate-200 dark:border-slate-700 p-6">
        <div class="flex flex-col lg:flex-row items-start lg:items-center justify-between gap-4">
            <div>
                <h1 class="text-2xl font-bold">Analytics Dashboard</h1>
                <p class="text-sm text-slate-600 dark:text-slate-400 mt-1">Key KPIs and trends</p>
            </div>
            <div class="flex flex-wrap items-center gap-3">
                <div class="flex bg-slate-100 dark:bg-slate-800 rounded-lg p-1">
                    <button type="button" data-period="7d" class="period-btn px-3 py-2 text-sm rounded-md hover:bg-white dark:hover:bg-slate-700">7 Days</button>
                    <button type="button" data-period="30d" class="period-btn px-3 py-2 text-sm rounded-md bg-indigo-600 text-white">30 Days</button>
                    <button type="button" data-period="90d" class="period-btn px-3 py-2 text-sm rounded-md hover:bg-white dark:hover:bg-slate-700">90 Days</button>
                    <button type="button" data-period="1y" class="period-btn px-3 py-2 text-sm rounded-md hover:bg-white dark:hover:bg-slate-700">1 Year</button>
                </div>
                <label class="flex items-center gap-2 text-sm">
                    <input type="checkbox" id="autoRefresh" class="rounded border-slate-300 dark:border-slate-600" checked>
                    <span>Auto-refresh</span>
                </label>
                <div class="text-xs text-slate-500 dark:text-slate-400">Last updated: <span id="lastUpdated" class="font-medium">—</span></div>
            </div>
        </div>
    </div>

    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6">
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">Today's Revenue</p>
            <p class="text-2xl font-bold" id="todayRevenue">₦0</p>
            <p class="text-sm"><span id="revenueGrowth" class="text-green-600 dark:text-green-400">+0%</span> <span class="text-slate-500">vs yesterday</span></p>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">Today's Orders</p>
            <p class="text-2xl font-bold" id="todayOrders">0</p>
            <p class="text-sm text-slate-500">Pending: <span id="pendingOrders" class="text-orange-600">0</span></p>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">Avg Order Value</p>
            <p class="text-2xl font-bold" id="avgOrderValue">₦0</p>
            <p class="text-sm text-slate-500">This period</p>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">Customer Retention</p>
            <p class="text-2xl font-bold" id="retentionRate">0%</p>
            <p class="text-sm text-slate-500">Repeat customers</p>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <p class="text-sm text-slate-600 dark:text-slate-400">New Customers</p>
            <p class="text-2xl font-bold" id="newCustomers">0</p>
            <p class="text-sm text-slate-500">Today</p>
        </div>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Revenue Trends</h2>
            <div class="h-64"><canvas id="revenueChart"></canvas></div>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Revenue by Category</h2>
            <div class="h-64"><canvas id="categoryChart"></canvas></div>
        </div>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Top Customers</h2>
            <div id="topCustomers" class="space-y-2">
                <div class="h-4 bg-slate-200 dark:bg-slate-700 rounded w-3/4"></div>
                <div class="h-4 bg-slate-200 dark:bg-slate-700 rounded w-1/2"></div>
            </div>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Best Sellers</h2>
            <div class="h-64"><canvas id="productsChart"></canvas></div>
        </div>
        <div class="surface rounded-lg p-6 border border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-900">
            <h2 class="text-lg font-semibold mb-4">Recent Activity</h2>
            <div id="recentActivity" class="space-y-2">
                <div class="h-4 bg-slate-200 dark:bg-slate-700 rounded"></div>
                <div class="h-4 bg-slate-200 dark:bg-slate-700 rounded w-3/4"></div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
(function(){
    'use strict';
    const base = '{{ url('/super-admin/analytics') }}';
    const fmt = (n) => Number(n||0).toLocaleString('en-NG', { minimumFractionDigits:2, maximumFractionDigits:2 });
    const qsAll = (sel) => Array.from(document.querySelectorAll(sel));
    const state = { period: '30d', charts: {}, timer: null };

    function setActive(btn){
        qsAll('.period-btn').forEach(b=>{ b.classList.remove('bg-indigo-600','text-white'); });
        btn.classList.add('bg-indigo-600','text-white');
    }
    function text(id, val){ const el = document.getElementById(id); if(el) el.textContent = val; }
    function now(){ const el=document.getElementById('lastUpdated'); if(el) el.textContent = new Date().toLocaleTimeString(); }
    function destroy(name){ if(state.charts[name]){ state.charts[name].destroy(); delete state.charts[name]; } }
    async function getJson(url){ const r = await fetch(url, { headers:{ 'X-Requested-With':'XMLHttpRequest' } }); if(!r.ok) throw new Error('HTTP '+r.status); return r.json(); }

    async function loadRealtime(){
        try{ const d = await getJson(`${base}/realtime`);
            text('todayRevenue', '₦'+fmt(d.today_metrics?.revenue_today||0));
            text('todayOrders', d.today_metrics?.orders_today||0);
            text('pendingOrders', d.today_metrics?.pending_orders||0);
            text('avgOrderValue', '₦'+fmt(d.today_metrics?.avg_order_value_today||0));
            text('newCustomers', d.today_metrics?.new_customers_today||0);
            if(Array.isArray(d.recent_orders)){
                const box = document.getElementById('recentActivity');
                if(box) box.innerHTML = d.recent_orders.slice(0,5).map(o=>`<div class="flex items-center justify-between text-sm py-2 border-b border-slate-200 dark:border-slate-700 last:border-0"><span>#${o.id} - ${escapeHtml(o.user?.name||'Guest')}</span><span class="font-medium">₦${fmt(o.total||0)}</span></div>`).join('');
            }
        }catch(e){ console.error('Realtime failed', e); }
    }
    async function loadRevenue(){
        try{ const d = await getJson(`${base}/revenue?period=${encodeURIComponent(state.period)}`);
            const g = Number(d.revenue_growth||0); const el = document.getElementById('revenueGrowth');
            if(el){ el.textContent = (g>=0?'+':'')+g.toFixed(1)+'%'; el.className = (g>=0? 'text-green-600 dark:text-green-400':'text-red-600 dark:text-red-400')+' font-medium'; }
            if(Array.isArray(d.revenue_data)) renderRevenue(d.revenue_data);
            if(Array.isArray(d.category_revenue)) renderCategory(d.category_revenue);
        }catch(e){ console.error('Revenue failed', e); }
    }
    async function loadCustomers(){
        try{ const d = await getJson(`${base}/customers?period=${encodeURIComponent(state.period)}`);
            text('retentionRate', (Number(d.repeat_rate||0)).toFixed(1)+'%');
            if(Array.isArray(d.customer_ltv)){
                const box=document.getElementById('topCustomers');
                if(box) box.innerHTML = d.customer_ltv.slice(0,5).map(c=>`<div class="flex items-center justify-between py-2 border-b border-slate-200 dark:border-slate-700 last:border-0"><div><div class="font-medium">${escapeHtml(c.name||'—')}</div><div class="text-sm text-slate-500">${Number(c.total_orders||0)} orders</div></div><div class="font-medium">₦${fmt(c.lifetime_value||0)}</div></div>`).join('');
            }
        }catch(e){ console.error('Customers failed', e); }
    }
    async function loadProducts(){
        try{ const d = await getJson(`${base}/products?period=${encodeURIComponent(state.period)}`);
            if(Array.isArray(d.best_sellers)) renderProducts(d.best_sellers);
        }catch(e){ console.error('Products failed', e); }
    }

    function renderRevenue(rows){ destroy('revenue'); const ctx=document.getElementById('revenueChart'); if(!ctx||!window.Chart) return; state.charts.revenue = new Chart(ctx,{ type:'line', data:{ labels: rows.map(r=> new Date(r.date).toLocaleDateString()), datasets:[{ label:'Revenue (₦)', data: rows.map(r=> Number(r.revenue)||0), borderColor:'#3B82F6', backgroundColor:'rgba(59,130,246,0.1)', fill:true, tension:0.35 }] }, options:{ responsive:true, maintainAspectRatio:false, scales:{ y:{ beginAtZero:true, ticks:{ callback:v=>'₦'+fmt(v) } } } } }); }
    function renderCategory(rows){ destroy('category'); const ctx=document.getElementById('categoryChart'); if(!ctx||!window.Chart) return; state.charts.category = new Chart(ctx,{ type:'doughnut', data:{ labels: rows.map(r=> r.name), datasets:[{ data: rows.map(r=> Number(r.revenue)||0), backgroundColor:['#3B82F6','#10B981','#F59E0B','#EF4444','#8B5CF6','#06B6D4','#84CC16','#F97316','#EC4899','#6366F1'] }] }, options:{ responsive:true, maintainAspectRatio:false } }); }
    function renderProducts(rows){ destroy('products'); const ctx=document.getElementById('productsChart'); if(!ctx||!window.Chart) return; const top=rows.slice(0,5); state.charts.products = new Chart(ctx,{ type:'bar', data:{ labels: top.map(r=> r.name.length>14 ? r.name.slice(0,14)+'…' : r.name), datasets:[{ label:'Units', data: top.map(r=> Number(r.total_sold)||0), backgroundColor:'rgba(16,185,129,0.8)' }] }, options:{ responsive:true, maintainAspectRatio:false, plugins:{ legend:{ display:false } }, scales:{ y:{ beginAtZero:true } } } }); }
    function escapeHtml(t){ const d=document.createElement('div'); d.textContent=String(t??''); return d.innerHTML; }

    async function refreshAll(){ await Promise.all([loadRealtime(), loadRevenue(), loadCustomers(), loadProducts()]); now(); }
    function bind(){
        qsAll('.period-btn').forEach(btn=> btn.addEventListener('click',()=>{ state.period = btn.getAttribute('data-period'); setActive(btn); refreshAll(); }));
        const ar=document.getElementById('autoRefresh');
        if(ar){ const stop=()=>{ if(state.timer){ clearInterval(state.timer); state.timer=null; } }; const start=()=>{ stop(); state.timer=setInterval(()=> loadRealtime().then(now).catch(()=>{}), 30000); }; ar.addEventListener('change', e=> e.target.checked? start() : stop()); start(); }
    }
    document.addEventListener('DOMContentLoaded', ()=>{ bind(); refreshAll(); });
})();
</script>
@endpush
