@extends('super-admin.layout')

@section('title', 'Performance Report')
@section('subtitle', $summary['period'])

@section('content')
<div class="space-y-6">
    <!-- Report Summary -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Performance Summary</h3>
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
            <div class="text-center p-4 bg-blue-500/10 rounded-lg">
                <div class="text-2xl font-bold text-blue-600 dark:text-blue-400">{{ number_format($summary['total_orders_processed']) }}</div>
                <div class="text-sm text-muted">Orders Processed</div>
            </div>
            <div class="text-center p-4 bg-green-500/10 rounded-lg">
                <div class="text-2xl font-bold text-green-600 dark:text-green-400">{{ number_format($summary['avg_processing_time'], 1) }}m</div>
                <div class="text-sm text-muted">Avg Processing Time</div>
            </div>
            <div class="text-center p-4 bg-purple-500/10 rounded-lg">
                <div class="text-2xl font-bold text-purple-600 dark:text-purple-400">{{ number_format($summary['completion_rate'], 1) }}%</div>
                <div class="text-sm text-muted">Completion Rate</div>
            </div>
            <div class="text-center p-4 bg-red-500/10 rounded-lg">
                <div class="text-2xl font-bold text-red-600 dark:text-red-400">{{ number_format($summary['cancellation_rate'], 1) }}%</div>
                <div class="text-sm text-muted">Cancellation Rate</div>
            </div>
        </div>
        
        <!-- Additional Metrics -->
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mt-6">
            <div class="text-center p-4 border border-soft rounded-lg">
                <div class="text-xl font-bold text-primary">{{ $summary['peak_hour'] }}</div>
                <div class="text-sm text-muted">Peak Hour</div>
            </div>
            <div class="text-center p-4 border border-soft rounded-lg">
                <div class="text-xl font-bold text-primary">{{ $summary['busiest_day'] }}</div>
                <div class="text-sm text-muted">Busiest Day</div>
            </div>
        </div>
    </div>
    
    <!-- Export Options -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Export Options</h3>
        <div class="flex flex-wrap gap-3">
            <a href="{{ request()->fullUrlWithQuery(['format' => 'pdf']) }}" 
               class="btn btn-outline text-red-600 border-red-300 hover:bg-red-50 dark:hover:bg-red-900/20">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                Download PDF
            </a>
            <a href="{{ request()->fullUrlWithQuery(['format' => 'csv']) }}" 
               class="btn btn-outline text-green-600 border-green-300 hover:bg-green-50 dark:hover:bg-green-900/20">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 17v-2m3 2v-4m3 4v-6m2 10H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                Download CSV
            </a>
        </div>
    </div>
    
    <!-- Charts -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <!-- Peak Hours Chart -->
        <div class="surface rounded-xl p-6">
            <h3 class="text-lg font-semibold text-primary mb-4">Orders by Hour</h3>
            <div class="h-64">
                <canvas id="peakHoursChart"></canvas>
            </div>
        </div>
        
        <!-- Day of Week Analysis -->
        <div class="surface rounded-xl p-6">
            <h3 class="text-lg font-semibold text-primary mb-4">Orders by Day of Week</h3>
            <div class="h-64">
                <canvas id="dayOfWeekChart"></canvas>
            </div>
        </div>
    </div>
    
    <!-- Order Status Distribution -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Order Status Distribution</h3>
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
            @foreach($statusDistribution as $status => $count)
            <div class="text-center p-4 border border-soft rounded-lg">
                <div class="text-xl font-bold 
                    {{ $status === 'completed' ? 'text-green-600' : 
                       ($status === 'cancelled' ? 'text-red-600' : 
                       ($status === 'pending' ? 'text-yellow-600' : 'text-blue-600')) }}">
                    {{ number_format($count) }}
                </div>
                <div class="text-sm text-muted capitalize">{{ str_replace('_', ' ', $status) }}</div>
                @if($summary['total_orders_processed'] > 0)
                <div class="text-xs text-muted">{{ number_format(($count / $summary['total_orders_processed']) * 100, 1) }}%</div>
                @endif
            </div>
            @endforeach
        </div>
    </div>
    
    <!-- Performance Metrics -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Detailed Performance Metrics</h3>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
            <div class="text-center p-4 bg-green-500/10 rounded-lg">
                <div class="text-lg font-bold text-green-600 dark:text-green-400">{{ number_format($satisfactionMetrics['fastest_processing_time'], 1) }}m</div>
                <div class="text-sm text-muted">Fastest Processing</div>
            </div>
            <div class="text-center p-4 bg-yellow-500/10 rounded-lg">
                <div class="text-lg font-bold text-yellow-600 dark:text-yellow-400">{{ number_format($satisfactionMetrics['average_processing_time'], 1) }}m</div>
                <div class="text-sm text-muted">Average Processing</div>
            </div>
            <div class="text-center p-4 bg-red-500/10 rounded-lg">
                <div class="text-lg font-bold text-red-600 dark:text-red-400">{{ number_format($satisfactionMetrics['slowest_processing_time'], 1) }}m</div>
                <div class="text-sm text-muted">Slowest Processing</div>
            </div>
        </div>
    </div>
    
    <!-- Peak Hours Analysis -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Hourly Performance Analysis</h3>
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead>
                    <tr class="border-b border-soft">
                        <th class="text-left py-3 text-primary">Hour</th>
                        <th class="text-right py-3 text-primary">Orders</th>
                        <th class="text-right py-3 text-primary">Revenue</th>
                        <th class="text-right py-3 text-primary">Avg Order Value</th>
                        <th class="text-right py-3 text-primary">Performance</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($peakHours as $hour)
                    <tr class="border-b border-soft/50">
                        <td class="py-3 font-medium text-primary">{{ $hour['hour'] }}</td>
                        <td class="py-3 text-right text-primary">{{ number_format($hour['order_count']) }}</td>
                        <td class="py-3 text-right text-primary">₦{{ number_format($hour['revenue'], 0) }}</td>
                        <td class="py-3 text-right text-muted">₦{{ number_format($hour['avg_order_value'], 0) }}</td>
                        <td class="py-3 text-right">
                            @php
                                $maxOrders = $peakHours->max('order_count');
                                $performance = $maxOrders > 0 ? ($hour['order_count'] / $maxOrders) * 100 : 0;
                            @endphp
                            <div class="flex items-center justify-end">
                                <div class="w-16 bg-gray-200 dark:bg-gray-700 rounded-full h-2 mr-2">
                                    <div class="bg-blue-600 h-2 rounded-full" style="width: {{ $performance }}%"></div>
                                </div>
                                <span class="text-xs text-muted w-8">{{ number_format($performance, 0) }}%</span>
                            </div>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Day of Week Analysis -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Weekly Performance Breakdown</h3>
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead>
                    <tr class="border-b border-soft">
                        <th class="text-left py-3 text-primary">Day</th>
                        <th class="text-right py-3 text-primary">Orders</th>
                        <th class="text-right py-3 text-primary">Revenue</th>
                        <th class="text-right py-3 text-primary">Avg Order Value</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($dayOfWeekAnalysis as $day)
                    <tr class="border-b border-soft/50">
                        <td class="py-3 font-medium text-primary">{{ $day->day_name }}</td>
                        <td class="py-3 text-right text-primary">{{ number_format($day->order_count) }}</td>
                        <td class="py-3 text-right text-primary">₦{{ number_format($day->revenue, 0) }}</td>
                        <td class="py-3 text-right text-muted">₦{{ number_format($day->order_count > 0 ? $day->revenue / $day->order_count : 0, 0) }}</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Order Processing Details -->
    <div class="surface rounded-xl overflow-hidden">
        <div class="px-6 py-4 border-b border-soft">
            <h3 class="text-lg font-semibold text-primary">Order Processing Times</h3>
        </div>
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead class="bg-slate-50 dark:bg-slate-800">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-muted uppercase tracking-wider">Order ID</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-muted uppercase tracking-wider">Status</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-muted uppercase tracking-wider">Processing Time</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-muted uppercase tracking-wider">Created</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-muted uppercase tracking-wider">Updated</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-muted uppercase tracking-wider">Performance</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-soft">
                    @foreach($orderProcessingTimes->take(50) as $order)
                    <tr class="hover:bg-slate-50 dark:hover:bg-slate-800">
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-primary">#{{ $order['order_id'] }}</td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full 
                                {{ $order['status'] === 'completed' ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' : 
                                   ($order['status'] === 'cancelled' ? 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400' : 
                                   'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400') }}">
                                {{ ucfirst($order['status']) }}
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-right">
                            <span class="font-medium {{ $order['processing_time_minutes'] > 60 ? 'text-red-600' : 
                                                        ($order['processing_time_minutes'] > 30 ? 'text-orange-600' : 'text-green-600') }}">
                                {{ number_format($order['processing_time_minutes']) }}m
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-muted">{{ $order['created_at']->format('M d, H:i') }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-muted">{{ $order['updated_at']->format('M d, H:i') }}</td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            @if($order['processing_time_minutes'] <= 15)
                                <span class="px-2 py-1 text-xs font-semibold rounded-full bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400">Excellent</span>
                            @elseif($order['processing_time_minutes'] <= 30)
                                <span class="px-2 py-1 text-xs font-semibold rounded-full bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400">Good</span>
                            @elseif($order['processing_time_minutes'] <= 60)
                                <span class="px-2 py-1 text-xs font-semibold rounded-full bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400">Average</span>
                            @else
                                <span class="px-2 py-1 text-xs font-semibold rounded-full bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400">Slow</span>
                            @endif
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
        @if($orderProcessingTimes->count() > 50)
        <div class="px-6 py-3 bg-slate-50 dark:bg-slate-800 text-center text-sm text-muted">
            Showing first 50 orders of {{ $orderProcessingTimes->count() }} total orders
        </div>
        @endif
    </div>
</div>
@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// Peak Hours Chart
const peakHoursCtx = document.getElementById('peakHoursChart').getContext('2d');
const peakHoursData = @json($peakHours);
const hourLabels = peakHoursData.map(hour => hour.hour);
const hourOrders = peakHoursData.map(hour => hour.order_count);

new Chart(peakHoursCtx, {
    type: 'bar',
    data: {
        labels: hourLabels,
        datasets: [{
            label: 'Orders',
            data: hourOrders,
            backgroundColor: 'rgba(59, 130, 246, 0.8)',
            borderColor: 'rgb(59, 130, 246)',
            borderWidth: 1
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                display: false
            }
        },
        scales: {
            y: {
                beginAtZero: true
            }
        }
    }
});

// Day of Week Chart
const dayOfWeekCtx = document.getElementById('dayOfWeekChart').getContext('2d');
const dayOfWeekData = @json($dayOfWeekAnalysis);
const dayLabels = dayOfWeekData.map(day => day.day_name);
const dayOrders = dayOfWeekData.map(day => day.order_count);

new Chart(dayOfWeekCtx, {
    type: 'doughnut',
    data: {
        labels: dayLabels,
        datasets: [{
            data: dayOrders,
            backgroundColor: [
                'rgba(59, 130, 246, 0.8)',
                'rgba(16, 185, 129, 0.8)',
                'rgba(245, 158, 11, 0.8)',
                'rgba(239, 68, 68, 0.8)',
                'rgba(168, 85, 247, 0.8)',
                'rgba(6, 182, 212, 0.8)',
                'rgba(99, 102, 241, 0.8)'
            ],
            borderWidth: 2
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'bottom'
            }
        }
    }
});
</script>
@endpush