@extends('super-admin.layout')

@section('title', 'Sales Report')
@section('subtitle', $summary['period'])

@section('content')
<div class="space-y-6">
    <!-- Report Summary -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Sales Summary</h3>
        <div class="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
            <div class="text-center p-4 bg-green-500/10 rounded-lg">
                <div class="text-2xl font-bold text-green-600 dark:text-green-400">{{ number_format($summary['total_orders']) }}</div>
                <div class="text-sm text-muted">Total Orders</div>
            </div>
            <div class="text-center p-4 bg-blue-500/10 rounded-lg">
                <div class="text-2xl font-bold text-blue-600 dark:text-blue-400">{{ number_format($summary['completed_orders']) }}</div>
                <div class="text-sm text-muted">Completed</div>
            </div>
            <div class="text-center p-4 bg-orange-500/10 rounded-lg">
                <div class="text-2xl font-bold text-orange-600 dark:text-orange-400">{{ number_format($summary['pending_orders']) }}</div>
                <div class="text-sm text-muted">Pending</div>
            </div>
            <div class="text-center p-4 bg-red-500/10 rounded-lg">
                <div class="text-2xl font-bold text-red-600 dark:text-red-400">{{ number_format($summary['cancelled_orders']) }}</div>
                <div class="text-sm text-muted">Cancelled</div>
            </div>
            <div class="text-center p-4 bg-purple-500/10 rounded-lg">
                <div class="text-2xl font-bold text-purple-600 dark:text-purple-400">₦{{ number_format($summary['total_revenue'], 0) }}</div>
                <div class="text-sm text-muted">Total Revenue</div>
            </div>
            <div class="text-center p-4 bg-indigo-500/10 rounded-lg">
                <div class="text-2xl font-bold text-indigo-600 dark:text-indigo-400">₦{{ number_format($summary['avg_order_value'], 0) }}</div>
                <div class="text-sm text-muted">Avg Order Value</div>
            </div>
        </div>
        
        <!-- Additional Metrics -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mt-6">
            <div class="text-center p-4 border border-soft rounded-lg">
                <div class="text-xl font-bold text-primary">{{ number_format($summary['total_items_sold']) }}</div>
                <div class="text-sm text-muted">Items Sold</div>
            </div>
            <div class="text-center p-4 border border-soft rounded-lg">
                <div class="text-xl font-bold text-primary">{{ number_format($summary['conversion_rate'], 1) }}%</div>
                <div class="text-sm text-muted">Conversion Rate</div>
            </div>
            <div class="text-center p-4 border border-soft rounded-lg">
                <div class="text-xl font-bold text-primary">{{ $summary['days_in_period'] }}</div>
                <div class="text-sm text-muted">Days in Period</div>
            </div>
        </div>
    </div>
    
    <!-- Export Options -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Export Options</h3>
        <div class="flex flex-wrap gap-3">
            <button disabled
               class="btn btn-outline text-gray-400 border-gray-300 cursor-not-allowed opacity-50">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                PDF (Unavailable)
            </button>
            <a href="{{ request()->fullUrlWithQuery(['format' => 'csv']) }}" 
               class="btn btn-outline text-green-600 border-green-300 hover:bg-green-50 dark:hover:bg-green-900/20">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 17v-2m3 2v-4m3 4v-6m2 10H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                </svg>
                Download CSV
            </a>
            <button onclick="window.print()" 
                    class="btn btn-outline text-gray-600 border-gray-300 hover:bg-gray-50 dark:hover:bg-gray-900/20">
                <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 17h2a2 2 0 002-2v-4a2 2 0 00-2-2H5a2 2 0 00-2 2v4a2 2 0 002 2h2m2 4h6a2 2 0 002-2v-4a2 2 0 00-2-2H9a2 2 0 00-2 2v4a2 2 0 002 2zm8-12V5a2 2 0 00-2-2H9a2 2 0 00-2 2v4h10z"></path>
                </svg>
                Print Report
            </button>
        </div>
        <div class="mt-3 text-sm text-gray-500">
            <p><strong>Note:</strong> PDF export is temporarily unavailable. Use CSV export or print functionality as alternatives.</p>
        </div>
    </div>
    
    <!-- Charts and Analytics -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <!-- Daily Sales Chart -->
        <div class="surface rounded-xl p-6">
            <h3 class="text-lg font-semibold text-primary mb-4">Daily Sales Trend</h3>
            <div class="h-64">
                <canvas id="dailySalesChart"></canvas>
            </div>
        </div>
        
        <!-- Hourly Distribution -->
        <div class="surface rounded-xl p-6">
            <h3 class="text-lg font-semibold text-primary mb-4">Order Distribution by Hour</h3>
            <div class="h-64">
                <canvas id="hourlyChart"></canvas>
            </div>
        </div>
    </div>
    
    <!-- Category Performance -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Revenue by Category</h3>
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead>
                    <tr class="border-b border-soft">
                        <th class="text-left py-3 text-primary">Category</th>
                        <th class="text-right py-3 text-primary">Revenue</th>
                        <th class="text-right py-3 text-primary">Items Sold</th>
                        <th class="text-right py-3 text-primary">Orders</th>
                        <th class="text-right py-3 text-primary">Avg Price</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($categoryRevenue as $category)
                    <tr class="border-b border-soft/50">
                        <td class="py-3 font-medium text-primary">{{ $category->name }}</td>
                        <td class="py-3 text-right text-primary">₦{{ number_format($category->revenue, 0) }}</td>
                        <td class="py-3 text-right text-muted">{{ number_format($category->items_sold) }}</td>
                        <td class="py-3 text-right text-muted">{{ number_format($category->orders_count) }}</td>
                        <td class="py-3 text-right text-muted">₦{{ number_format($category->revenue / $category->items_sold, 0) }}</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Top Selling Dishes -->
    <div class="surface rounded-xl p-6">
        <h3 class="text-lg font-semibold text-primary mb-4">Top Selling Dishes</h3>
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead>
                    <tr class="border-b border-soft">
                        <th class="text-left py-3 text-primary">Dish</th>
                        <th class="text-left py-3 text-primary">Category</th>
                        <th class="text-right py-3 text-primary">Price</th>
                        <th class="text-right py-3 text-primary">Quantity Sold</th>
                        <th class="text-right py-3 text-primary">Revenue</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($topDishes as $dish)
                    <tr class="border-b border-soft/50">
                        <td class="py-3 font-medium text-primary">{{ $dish->name }}</td>
                        <td class="py-3 text-muted">{{ $dish->category }}</td>
                        <td class="py-3 text-right text-muted">₦{{ number_format($dish->price, 0) }}</td>
                        <td class="py-3 text-right text-primary font-medium">{{ number_format($dish->total_sold) }}</td>
                        <td class="py-3 text-right text-primary font-medium">₦{{ number_format($dish->total_revenue, 0) }}</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Orders Table -->
    <div class="surface rounded-xl overflow-hidden">
        <div class="px-6 py-4 border-b border-soft">
            <h3 class="text-lg font-semibold text-primary">Recent Orders</h3>
        </div>
        <div class="overflow-x-auto">
            <table class="w-full">
                <thead class="bg-slate-50 dark:bg-slate-800">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-muted uppercase tracking-wider">Order ID</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-muted uppercase tracking-wider">Customer</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-muted uppercase tracking-wider">Status</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-muted uppercase tracking-wider">Total</th>
                        <th class="px-6 py-3 text-right text-xs font-medium text-muted uppercase tracking-wider">Items</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-muted uppercase tracking-wider">Date</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-soft">
                    @foreach($orders->take(50) as $order)
                    <tr class="hover:bg-slate-50 dark:hover:bg-slate-800">
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-primary">#{{ $order->id }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-primary">
                            {{ $order->user->name ?? 'Guest' }}
                            @if($order->user)
                                <div class="text-xs text-muted">{{ $order->user->email }}</div>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full 
                                {{ $order->status === 'completed' ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' : 
                                   ($order->status === 'cancelled' ? 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400' : 
                                   'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400') }}">
                                {{ ucfirst($order->status) }}
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-right text-primary font-medium">₦{{ number_format($order->total, 0) }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-right text-muted">{{ optional($order->items)->sum('quantity') ?? 0 }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-muted">{{ $order->created_at->format('M d, Y H:i') }}</td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
        @if($orders->count() > 50)
        <div class="px-6 py-3 bg-slate-50 dark:bg-slate-800 text-center text-sm text-muted">
            Showing first 50 orders of {{ $orders->count() }} total orders
        </div>
        @endif
    </div>
</div>
@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// Daily Sales Chart
const dailySalesCtx = document.getElementById('dailySalesChart').getContext('2d');
const dailySalesData = @json($dailySales);
const dailyLabels = Object.keys(dailySalesData);
const dailyRevenue = Object.values(dailySalesData).map(day => day.revenue);
const dailyOrders = Object.values(dailySalesData).map(day => day.orders);

new Chart(dailySalesCtx, {
    type: 'line',
    data: {
        labels: dailyLabels,
        datasets: [{
            label: 'Revenue (₦)',
            data: dailyRevenue,
            borderColor: 'rgb(59, 130, 246)',
            backgroundColor: 'rgba(59, 130, 246, 0.1)',
            yAxisID: 'y'
        }, {
            label: 'Orders',
            data: dailyOrders,
            borderColor: 'rgb(16, 185, 129)',
            backgroundColor: 'rgba(16, 185, 129, 0.1)',
            yAxisID: 'y1'
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'top',
            }
        },
        scales: {
            y: {
                type: 'linear',
                display: true,
                position: 'left',
            },
            y1: {
                type: 'linear',
                display: true,
                position: 'right',
                grid: {
                    drawOnChartArea: false,
                }
            }
        }
    }
});

// Hourly Distribution Chart
const hourlyCtx = document.getElementById('hourlyChart').getContext('2d');
const hourlyData = @json($hourlyDistribution);
const hourlyLabels = Object.keys(hourlyData);
const hourlyOrders = Object.values(hourlyData).map(hour => hour.orders);

new Chart(hourlyCtx, {
    type: 'bar',
    data: {
        labels: hourlyLabels,
        datasets: [{
            label: 'Orders',
            data: hourlyOrders,
            backgroundColor: 'rgba(168, 85, 247, 0.8)',
            borderColor: 'rgb(168, 85, 247)',
            borderWidth: 1
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                display: false
            }
        },
        scales: {
            y: {
                beginAtZero: true
            }
        }
    }
});
</script>
@endpush