<?php

use App\Http\Controllers\ProfileController;
use App\Http\Controllers\MenuController;
use App\Http\Controllers\CartController;
use App\Http\Controllers\CheckoutController;
use App\Http\Controllers\CustomerOrderController;
use App\Http\Controllers\PaystackController;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\CustomerDashboardController;
use App\Http\Controllers\Admin\AdminAnalyticsController;
use App\Http\Controllers\LandingController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\DishController;
use App\Http\Controllers\OrderController;
use App\Http\Controllers\Admin\SettingController;
use App\Http\Controllers\ContactController;
use App\Http\Controllers\CustomerAddressController;
use App\Http\Controllers\Admin\CustomerAddressAdminController;
use Illuminate\Http\Request;
use App\Http\Controllers\Api\MeKpiController;

// Landing page
Route::get('/', [LandingController::class, 'index'])->name('home');

// Static pages (migrated from legacy .html)
Route::view('/privacy', 'privacy')->name('privacy');
Route::view('/terms', 'terms')->name('terms');
Route::view('/events', 'events')->name('events');
// Back-compat redirects from legacy filenames
Route::redirect('/privacy.html', '/privacy', 301);
Route::redirect('/terms.html', '/terms', 301);
Route::redirect('/events.html', '/events', 301);

// Public menu
Route::get('/menu', [MenuController::class, 'index'])->name('menu.index');
Route::get('/menu.json', [MenuController::class, 'json'])->name('menu.json');

// Contact
Route::get('/contact', [ContactController::class, 'show'])->name('contact.show');
Route::post('/contact', [ContactController::class, 'submit'])->middleware('throttle:10,1')->name('contact.submit');

// QR Code generation
Route::get('/qr', [App\Http\Controllers\QRCodeController::class, 'generate'])->name('qr.generate');
Route::get('/qr/pwa', [App\Http\Controllers\QRCodeController::class, 'pwaInstall'])->name('qr.pwa');
Route::get('/qr/contact', [App\Http\Controllers\QRCodeController::class, 'contact'])->name('qr.contact');

// Cart (session-based)
Route::get('/cart', [CartController::class, 'index'])->name('cart.index');
Route::post('/cart/add', [CartController::class, 'add'])->name('cart.add');
Route::post('/cart/update', [CartController::class, 'update'])->name('cart.update');
Route::post('/cart/remove', [CartController::class, 'remove'])->name('cart.remove');
Route::post('/cart/clear', [CartController::class, 'clear'])->name('cart.clear');

// Legacy compatibility: allow GET add via ?id= to support old links
Route::get('/cart/add', function (Request $request) {
    $id = (int) $request->query('id');
    if ($id <= 0 || !\App\Models\Dish::whereKey($id)->exists()) {
        return redirect()->route('menu.index')->with('status', 'Item not found');
    }
    $cart = session('cart', ['items' => []]);
    $cart['items'][$id] = ($cart['items'][$id] ?? 0) + 1;
    session(['cart' => $cart]);
    return redirect()->route('cart.index')->with('status', 'Added to cart');
})->name('cart.add.legacy');

Route::get('/dashboard', CustomerDashboardController::class)
    ->middleware(['auth'])
    ->name('dashboard');

Route::middleware('auth')->group(function () {
    Route::get('/profile', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::delete('/profile', [ProfileController::class, 'destroy'])->name('profile.destroy');

    // Checkout (feature gated)
    Route::middleware('feature:checkout')->group(function () {
        Route::get('/checkout', [CheckoutController::class, 'index'])->name('checkout.index');
        Route::post('/checkout', [CheckoutController::class, 'store'])->middleware('orders.capacity')->name('checkout.store');
    });

    // Customer orders (feature gated)
    Route::middleware('feature:orders')->group(function () {
        Route::get('/orders', [CustomerOrderController::class, 'index'])->name('orders.index');
        Route::get('/orders/{order}', [CustomerOrderController::class, 'show'])->name('orders.show');
        Route::get('/orders/{order}/status.json', [CustomerOrderController::class, 'statusJson'])->name('orders.status.json');
    });

    // Customer KPIs (lightweight JSON for dashboard tiles)
    Route::get('/api/me/kpis', MeKpiController::class)->name('api.me.kpis');

    // Customer addresses (feature gated)
    Route::middleware('feature:addresses')->group(function () {
        Route::get('/account/addresses', [CustomerAddressController::class, 'index'])->name('account.addresses.index');
        Route::post('/account/addresses', [CustomerAddressController::class, 'store'])->name('account.addresses.store');
        Route::post('/account/addresses/{address}/primary', [CustomerAddressController::class, 'setPrimary'])->name('account.addresses.primary');
        Route::delete('/account/addresses/{address}', [CustomerAddressController::class, 'destroy'])->name('account.addresses.destroy');
    });

    // Paystack callback
    Route::get('/paystack/callback', [PaystackController::class, 'callback'])->name('paystack.callback');
});

// Paystack Webhook (no auth, no CSRF; verify signature in controller)
use Illuminate\Foundation\Http\Middleware\VerifyCsrfToken as FrameworkVerifyCsrf;
Route::post('/paystack/webhook', [PaystackController::class, 'webhook'])
    ->withoutMiddleware([FrameworkVerifyCsrf::class])
    ->name('paystack.webhook');

require __DIR__.'/auth.php';

// Admin routes (RBAC)
Route::middleware(['auth', 'role:Super Admin|Manager|Kitchen|Cashier|Content Editor'])
    ->prefix('admin')
    ->as('admin.')
    ->group(function () {
        Route::get('/', AdminAnalyticsController::class)->name('dashboard');
    // Settings
    Route::get('settings', [SettingController::class, 'index'])->name('settings.index');
    Route::put('settings', [SettingController::class, 'update'])->name('settings.update');
        Route::resource('categories', CategoryController::class)->except(['show']);
    Route::resource('dishes', DishController::class)->except(['show']);
    Route::get('dishes-trash', [DishController::class,'trash'])->name('dishes.trash');
    Route::post('dishes/{id}/restore', [DishController::class,'restore'])->name('dishes.restore');
    Route::delete('dishes/{id}/force', [DishController::class,'forceDelete'])->name('dishes.force-delete');
        Route::get('orders', [OrderController::class, 'index'])->name('orders.index');
        Route::get('orders/{order}', [OrderController::class, 'show'])->name('orders.show');
    Route::get('orders/{order}/print', [OrderController::class, 'print'])->name('orders.print');
    Route::get('orders-rows', [OrderController::class, 'rows'])->name('orders.rows');
    Route::get('orders-export', [OrderController::class, 'export'])->name('orders.export');
    Route::post('orders/bulk-update', [OrderController::class, 'bulkUpdate'])->name('orders.bulk-update');
        Route::patch('orders/{order}', [OrderController::class, 'update'])->name('orders.update');

    // Inventory
    Route::get('inventory', [\App\Http\Controllers\Admin\InventoryController::class, 'index'])->name('inventory.index');
    Route::get('inventory/movements', [\App\Http\Controllers\Admin\InventoryController::class, 'movements'])->name('inventory.movements');
    Route::get('inventory/movements/export', [\App\Http\Controllers\Admin\InventoryController::class, 'exportMovements'])->name('inventory.movements.export');
    // Inventory availability per location
    Route::get('inventory/availability', [\App\Http\Controllers\Admin\InventoryController::class, 'availability'])->name('inventory.availability');
    Route::get('inventory/availability/pivot', [\App\Http\Controllers\Admin\InventoryController::class, 'availabilityPivot'])->name('inventory.availability.pivot');
    Route::get('inventory/availability/export', [\App\Http\Controllers\Admin\InventoryController::class, 'exportAvailability'])->name('inventory.availability.export');
    Route::get('inventory/availability/totals', [\App\Http\Controllers\Admin\InventoryController::class, 'availabilityTotals'])->name('inventory.availability.totals');
    Route::get('inventory/availability/sparkline', [\App\Http\Controllers\Admin\InventoryController::class, 'availabilitySparkline'])->name('inventory.availability.sparkline');
    // Transfer stock between locations (admin)
    Route::get('inventory/transfer', [\App\Http\Controllers\Admin\InventoryController::class, 'transferForm'])->name('inventory.transfer.form');
    Route::post('inventory/transfer', [\App\Http\Controllers\Admin\InventoryController::class, 'transferStore'])->name('inventory.transfer.store');

    // Locations admin CRUD
    Route::resource('locations', \App\Http\Controllers\Admin\LocationController::class)->except(['show']);
    Route::post('dishes/{dish}/inventory/adjust', [\App\Http\Controllers\Admin\InventoryController::class, 'adjust'])->name('dishes.inventory.adjust');
    Route::get('dishes/{dish}/inventory', [\App\Http\Controllers\Admin\InventoryController::class, 'history'])->name('dishes.inventory.history');

    // Reporting & KPIs
    Route::get('reporting', [\App\Http\Controllers\Admin\ReportingController::class, 'index'])->name('reporting.index');
    Route::get('reporting/api', [\App\Http\Controllers\Admin\ReportingController::class, 'api'])->name('reporting.api');
    Route::get('reporting/export', [\App\Http\Controllers\Admin\ReportingController::class, 'export'])->name('reporting.export');
    Route::post('reporting/clear-cache', [\App\Http\Controllers\Admin\ReportingController::class, 'clearCache'])->name('reporting.clear-cache');

    // Contact messages
    Route::get('contact-messages', [ContactController::class, 'index'])->name('contact-messages.index');
        // Admin customer addresses
        Route::get('addresses', [CustomerAddressAdminController::class, 'index'])->name('addresses.index');
        Route::get('addresses/{address}', [CustomerAddressAdminController::class, 'show'])->name('addresses.show');
    });

// Local-only debug endpoint to confirm DB in use and table list
if (app()->environment('local')) {
    // Basic health check (lightweight, safe for uptime monitors). Returns 200 if app OK.
    Route::get('/healthz', function () {
        $start = microtime(true);
        $dbOk = true; $dbError = null;
        try {
            \Illuminate\Support\Facades\DB::select('SELECT 1');
        } catch (\Throwable $e) {
            $dbOk = false; $dbError = $e->getMessage();
        }
        return response()->json([
            'app' => config('app.name'),
            'env' => config('app.env'),
            'time' => now()->toIso8601String(),
            'uptime_check' => true,
            'db' => [ 'ok' => $dbOk, 'error' => $dbError ],
            'duration_ms' => round((microtime(true)-$start)*1000,2)
        ], $dbOk ? 200 : 500);
    })->name('healthz');

    Route::get('/_debug/db', function () {
        $connection = config('database.default');
        $configDb   = config("database.connections.{$connection}.database");
        try {
            $currentDb = DB::select('select database() as db');
            $currentDbName = $currentDb[0]->db ?? null;
        } catch (\Throwable $e) {
            $currentDbName = null;
        }

        $tables = [];
        try {
            $tables = DB::select('SHOW TABLES');
        } catch (\Throwable $e) {
            // ignore
        }

        return response()->json([
            'env' => [
                'APP_ENV' => env('APP_ENV'),
                'DB_CONNECTION' => env('DB_CONNECTION'),
                'DB_DATABASE_env' => env('DB_DATABASE'),
            ],
            'config' => [
                'default_connection' => $connection,
                'database_config' => $configDb,
            ],
            'runtime' => [
                'database()' => $currentDbName,
                'tables' => array_map(function ($row) {
                    // Each row is an stdClass with a single property like "Tables_in_dbname"
                    return array_values((array) $row)[0] ?? null;
                }, $tables),
            ],
        ]);
    })->name('debug.db');
}

// Super Admin Routes (Requires Super Admin role)
Route::prefix('super-admin')->name('super-admin.')->middleware(['auth', 'super.admin'])->group(function () {
    Route::get('/', [App\Http\Controllers\SuperAdmin\SuperAdminDashboardController::class, 'index'])->name('dashboard');
    
    // User Management (Role Edit Only)
    Route::prefix('users')->name('users.')->group(function () {
        Route::get('/', [App\Http\Controllers\SuperAdmin\UserManagementController::class, 'index'])->name('index');
        Route::get('/{user}/edit', [App\Http\Controllers\SuperAdmin\UserManagementController::class, 'edit'])->name('edit');
        Route::put('/{user}', [App\Http\Controllers\SuperAdmin\UserManagementController::class, 'update'])->name('update');
    });
    
    // Analytics
    Route::prefix('analytics')->name('analytics.')->group(function () {
        Route::get('/', [App\Http\Controllers\SuperAdmin\AnalyticsController::class, 'index'])->name('index');
        Route::get('/revenue', [App\Http\Controllers\SuperAdmin\AnalyticsController::class, 'revenue'])->name('revenue');
        Route::get('/customers', [App\Http\Controllers\SuperAdmin\AnalyticsController::class, 'customers'])->name('customers');
        Route::get('/products', [App\Http\Controllers\SuperAdmin\AnalyticsController::class, 'products'])->name('products');
        Route::get('/operational', [App\Http\Controllers\SuperAdmin\AnalyticsController::class, 'operational'])->name('operational');
        Route::get('/financial', [App\Http\Controllers\SuperAdmin\AnalyticsController::class, 'financial'])->name('financial');
        Route::get('/realtime', [App\Http\Controllers\SuperAdmin\AnalyticsController::class, 'realtime'])->name('realtime');
    });
    
    // Reports
    Route::prefix('reports')->name('reports.')->group(function () {
        Route::get('/', [App\Http\Controllers\SuperAdmin\ReportsController::class, 'index'])->name('index');
        Route::get('/sales', [App\Http\Controllers\SuperAdmin\ReportsController::class, 'salesReport'])->name('sales');
        Route::get('/customers', [App\Http\Controllers\SuperAdmin\ReportsController::class, 'customerReport'])->name('customers');
        Route::get('/products', [App\Http\Controllers\SuperAdmin\ReportsController::class, 'productReport'])->name('products');
        Route::get('/inventory', [App\Http\Controllers\SuperAdmin\ReportsController::class, 'inventoryReport'])->name('inventory');
        Route::get('/performance', [App\Http\Controllers\SuperAdmin\ReportsController::class, 'performanceReport'])->name('performance');
    });
    
    // Include all regular admin routes with super-admin prefix for easy access
    Route::prefix('admin')->name('admin.')->group(function () {
        Route::get('/dashboard', [App\Http\Controllers\Admin\AdminAnalyticsController::class, 'index'])->name('dashboard');
        Route::resource('categories', CategoryController::class);
        Route::resource('dishes', DishController::class);
        Route::resource('orders', OrderController::class);
        Route::get('/settings', [SettingController::class, 'index'])->name('settings.index');
        Route::put('/settings', [SettingController::class, 'update'])->name('settings.update');
        Route::resource('customer-addresses', CustomerAddressAdminController::class);
    });
});
