import fs from 'fs';
import path from 'path';
import sharp from 'sharp';

const root = path.resolve(process.cwd());
const src = path.join(root, 'public', 'assets', 'logo.png');
const outDir = path.join(root, 'public', 'icons');

const sizes = [16, 32, 180, 192, 256, 384, 512];

async function ensureDir(dir) {
  await fs.promises.mkdir(dir, { recursive: true });
}

async function build() {
  if (!fs.existsSync(src)) {
    console.error(`Logo not found at ${src}`);
    process.exit(1);
  }
  await ensureDir(outDir);

  const outputs = [
    { name: 'favicon-16x16.png', size: 16 },
    { name: 'favicon-32x32.png', size: 32 },
    { name: 'apple-touch-icon.png', size: 180 },
    { name: 'android-chrome-192x192.png', size: 192 },
    { name: 'android-chrome-256x256.png', size: 256 },
    { name: 'android-chrome-384x384.png', size: 384 },
    { name: 'android-chrome-512x512.png', size: 512 },
    // Maskable variants (padded contain)
    { name: 'maskable-192x192.png', size: 192, maskable: true },
    { name: 'maskable-512x512.png', size: 512, maskable: true },
  ];

  for (const o of outputs) {
    const dest = path.join(outDir, o.name);
    const pipeline = sharp(src)
      .resize(o.size, o.size, { fit: 'contain', background: { r: 0, g: 0, b: 0, alpha: 0 } })
      .png();
    await pipeline.toFile(dest);
    console.log('Generated', dest);
  }

  console.log('All icons generated to', outDir);
}

build().catch((err) => {
  console.error(err);
  process.exit(1);
});
