#!/usr/bin/env node
/**
 * Generate responsive image variants and LQIP under public/assets/images.
 * Creates {name}-{w}.{ext} for w in [320,640,1024,1536] and webp/avif; plus {name}-lqip.jpg (~24px width).
 */
import fs from 'fs';
import path from 'path';
import sharp from 'sharp';

const ROOT = process.cwd();
const IMG_DIR = path.join(ROOT, 'public', 'assets', 'images');
const STORAGE_DIR = path.join(ROOT, 'public', 'storage');
const widths = [320, 640, 1024, 1536];

function isRaster(file) {
  return /\.(jpe?g|png)$/i.test(file);
}

async function processFile(filePath) {
  const ext = path.extname(filePath).toLowerCase();
  const base = path.basename(filePath, ext);
  const dir = path.dirname(filePath);
  const input = sharp(filePath).rotate();

  // LQIP (~24px width) as JPEG
  const lqip = path.join(dir, `${base}-lqip.jpg`);
  try { await input.clone().resize({ width: 24 }).jpeg({ quality: 35 }).toFile(lqip); } catch {}

  for (const w of widths) {
    const outJpg = path.join(dir, `${base}-${w}.jpg`);
    const outWebp = path.join(dir, `${base}-${w}.webp`);
    const outAvif = path.join(dir, `${base}-${w}.avif`);
    try { await input.clone().resize({ width: w }).jpeg({ quality: 78 }).toFile(outJpg); } catch {}
    try { await input.clone().resize({ width: w }).webp({ quality: 75 }).toFile(outWebp); } catch {}
    try { await input.clone().resize({ width: w }).avif({ quality: 50 }).toFile(outAvif); } catch {}
  }
}

async function walk(dir) {
  const entries = fs.readdirSync(dir, { withFileTypes: true });
  for (const e of entries) {
    const fp = path.join(dir, e.name);
    if (e.isDirectory()) await walk(fp);
    else if (e.isFile() && isRaster(e.name)) await processFile(fp);
  }
}

(async () => {
  let any = false;
  if (fs.existsSync(IMG_DIR)) {
    console.log('Generating responsive images in', IMG_DIR);
    await walk(IMG_DIR); any = true;
  } else {
    console.warn('Image dir not found:', IMG_DIR);
  }
  if (fs.existsSync(STORAGE_DIR)) {
    console.log('Generating responsive images in', STORAGE_DIR);
    await walk(STORAGE_DIR); any = true;
  } else {
    console.warn('Storage dir not found:', STORAGE_DIR);
  }
  if (!any) {
    console.error('No image directories found.');
    process.exit(1);
  }
  console.log('Done.');
})();
