#!/usr/bin/env node
/**
 * generate-variants.js
 * Advanced image variant generator for slider (hero) and dish images.
 * 1. Reads source images from ./resources/raw/slider and ./resources/raw/dishes
 * 2. Produces optimized WebP + AVIF + fallback JPEG outputs into public/assets/{slider|dishes}
 * 3. Generates a manifest JSON mapping for integration in Blade components.
 *
 * Slider target widths: [480,768,1024,1280,1536,1920]
 * Dish (grid/detail) widths: [200,400,600,800,1200,1600]
 * Thumbnails (dishes): 200 square crop and LQIP (blurred) placeholder.
 */
import fs from 'fs';
import path from 'path';
import sharp from 'sharp';

const ROOT = process.cwd();
const RAW_DIR = path.join(ROOT, 'resources', 'raw');
const RAW_SLIDER = path.join(RAW_DIR, 'slider');
const RAW_DISHES = path.join(RAW_DIR, 'dishes');
const OUT_SLIDER = path.join(ROOT, 'public', 'assets', 'slider');
const OUT_DISHES = path.join(ROOT, 'public', 'assets', 'dishes');
const MANIFEST = path.join(ROOT, 'public', 'assets', 'image-manifest.json');

const sliderWidths = [480,768,1024,1280,1536,1920];
const dishWidths = [200,400,600,800,1200,1600];

function ensureDir(dir) { if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true }); }
function isRaster(file) { return /\.(jpe?g|png)$/i.test(file); }

async function processSlider(file, manifest) {
  const baseName = path.basename(file, path.extname(file));
  const outBaseDir = OUT_SLIDER;
  ensureDir(outBaseDir);
  const img = sharp(file).rotate();
  const entry = { type: 'slider', base: baseName, variants: [] };
  for (const w of sliderWidths) {
    const webp = path.join(outBaseDir, `${baseName}-${w}.webp`);
    const avif = path.join(outBaseDir, `${baseName}-${w}.avif`);
    const jpg = path.join(outBaseDir, `${baseName}-${w}.jpg`);
    await img.clone().resize({ width: w }).webp({ quality: 75 }).toFile(webp).catch(()=>{});
    await img.clone().resize({ width: w }).avif({ quality: 50 }).toFile(avif).catch(()=>{});
    await img.clone().resize({ width: w }).jpeg({ quality: 78 }).toFile(jpg).catch(()=>{});
    entry.variants.push({ width: w, webp: path.relative(path.join(ROOT,'public'), webp).replace(/\\/g,'/'), avif: path.relative(path.join(ROOT,'public'), avif).replace(/\\/g,'/'), jpg: path.relative(path.join(ROOT,'public'), jpg).replace(/\\/g,'/') });
  }
  // LQIP placeholder (blurred tiny jpg)
  const lqip = path.join(outBaseDir, `${baseName}-lqip.jpg`);
  await img.clone().resize({ width: 32 }).blur(8).jpeg({ quality: 35 }).toFile(lqip).catch(()=>{});
  entry.lqip = path.relative(path.join(ROOT,'public'), lqip).replace(/\\/g,'/');
  manifest.slider.push(entry);
}

async function processDish(file, manifest) {
  const baseName = path.basename(file, path.extname(file));
  const outBaseDir = OUT_DISHES;
  ensureDir(outBaseDir);
  const img = sharp(file).rotate();
  const entry = { type: 'dish', base: baseName, variants: [], thumb: null, lqip: null };
  for (const w of dishWidths) {
    const webp = path.join(outBaseDir, `${baseName}-${w}.webp`);
    const avif = path.join(outBaseDir, `${baseName}-${w}.avif`);
    const jpg = path.join(outBaseDir, `${baseName}-${w}.jpg`);
    await img.clone().resize({ width: w }).webp({ quality: 75 }).toFile(webp).catch(()=>{});
    await img.clone().resize({ width: w }).avif({ quality: 50 }).toFile(avif).catch(()=>{});
    await img.clone().resize({ width: w }).jpeg({ quality: 80 }).toFile(jpg).catch(()=>{});
    entry.variants.push({ width: w, webp: path.relative(path.join(ROOT,'public'), webp).replace(/\\/g,'/'), avif: path.relative(path.join(ROOT,'public'), avif).replace(/\\/g,'/'), jpg: path.relative(path.join(ROOT,'public'), jpg).replace(/\\/g,'/') });
  }
  // Square thumbnail 200x200 crop center
  const thumb = path.join(outBaseDir, `${baseName}-thumb-200.jpg`);
  await img.clone().resize(200,200,{ fit: 'cover', position: 'centre' }).jpeg({ quality: 78 }).toFile(thumb).catch(()=>{});
  entry.thumb = path.relative(path.join(ROOT,'public'), thumb).replace(/\\/g,'/');
  const lqip = path.join(outBaseDir, `${baseName}-lqip.jpg`);
  await img.clone().resize({ width: 24 }).blur(6).jpeg({ quality: 30 }).toFile(lqip).catch(()=>{});
  entry.lqip = path.relative(path.join(ROOT,'public'), lqip).replace(/\\/g,'/');
  manifest.dishes.push(entry);
}

function collect(dir) {
  if (!fs.existsSync(dir)) return [];
  return fs.readdirSync(dir).filter(isRaster).map(f => path.join(dir,f));
}

async function main() {
  ensureDir(OUT_SLIDER); ensureDir(OUT_DISHES); ensureDir(path.dirname(MANIFEST));
  const manifest = { generatedAt: new Date().toISOString(), slider: [], dishes: [] };
  const sliderFiles = collect(RAW_SLIDER);
  const dishFiles = collect(RAW_DISHES);
  if (!sliderFiles.length && !dishFiles.length) {
    console.error('No source images found. Place files in resources/raw/slider or resources/raw/dishes');
    process.exit(1);
  }
  console.log('Processing slider images:', sliderFiles.length);
  for (const f of sliderFiles) await processSlider(f, manifest);
  console.log('Processing dish images:', dishFiles.length);
  for (const f of dishFiles) await processDish(f, manifest);
  fs.writeFileSync(MANIFEST, JSON.stringify(manifest, null, 2));
  console.log('Manifest written to', MANIFEST);
  console.log('Done.');
}

main().catch(e => { console.error(e); process.exit(1); });
