<?php
// Compare two MySQL databases (schemas + row counts) and print a diff
// Usage: php scripts/compare_dbs.php [dbA] [dbB]
// Defaults: dbA=bonny_delight, dbB=bonny_delight_laravel

error_reporting(E_ALL);
ini_set('display_errors', '1');

$dbA = $argv[1] ?? 'bonny_delight';
$dbB = $argv[2] ?? 'bonny_delight_laravel';

$host = '127.0.0.1';
$user = 'root';
$pass = 'vick247$';

function snapshot(PDO $pdo, string $db): array {
    $pdo->exec("USE `{$db}`");
    $tables = $pdo->query('SHOW TABLES')->fetchAll(PDO::FETCH_COLUMN);
    $info = ['tables' => [], 'columns' => []];
    foreach ($tables as $t) {
        // Row count (may be approximate for InnoDB but good enough here)
        try {
            $cnt = (int) $pdo->query("SELECT COUNT(*) FROM `{$t}`")->fetchColumn();
        } catch (Throwable $e) {
            $cnt = -1;
        }
        $info['tables'][$t] = $cnt;
        // Columns
        try {
            $cols = $pdo->query("SHOW COLUMNS FROM `{$t}`")->fetchAll(PDO::FETCH_ASSOC);
            $normalized = [];
            foreach ($cols as $c) {
                $normalized[$c['Field']] = [
                    'type' => strtolower($c['Type']),
                    'null' => $c['Null'] === 'YES',
                    'key' => $c['Key'],
                    'default' => $c['Default'],
                    'extra' => $c['Extra'],
                ];
            }
            $info['columns'][$t] = $normalized;
        } catch (Throwable $e) {
            // Table may be malformed or missing; record as unknown
            $info['columns'][$t] = ['__error' => $e->getMessage()];
        }
    }
    return $info;
}

try {
    $pdo = new PDO("mysql:host={$host};charset=utf8mb4", $user, $pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
} catch (Throwable $e) {
    fwrite(STDERR, "Connection failed: {$e->getMessage()}\n");
    exit(1);
}

function array_diff_keys_only(array $a, array $b): array {
    return array_values(array_diff(array_keys($a), array_keys($b)));
}

echo "Reading schemas...\n";
$a = snapshot($pdo, $dbA);
$b = snapshot($pdo, $dbB);

$tablesA = $a['tables'];
$tablesB = $b['tables'];

$onlyA = array_diff_keys_only($tablesA, $tablesB);
$onlyB = array_diff_keys_only($tablesB, $tablesA);
$both  = array_values(array_intersect(array_keys($tablesA), array_keys($tablesB)));

echo "\nDatabases: A={$dbA}  B={$dbB}\n";
echo "Tables in A only (candidates to keep as legacy or port to B):\n";
foreach ($onlyA as $t) {
    printf("  - %-30s (%d rows)\n", $t, $tablesA[$t]);
}
echo "\nTables in B only (candidates to bring into A if valuable):\n";
foreach ($onlyB as $t) {
    printf("  + %-30s (%d rows)\n", $t, $tablesB[$t]);
}

echo "\nTables present in both with column diffs:\n";
foreach ($both as $t) {
    $colsA = $a['columns'][$t];
    $colsB = $b['columns'][$t];
    $colsAOnly = array_diff(array_keys($colsA), array_keys($colsB));
    $colsBOnly = array_diff(array_keys($colsB), array_keys($colsA));
    $changed = [];
    foreach (array_intersect(array_keys($colsA), array_keys($colsB)) as $c) {
        if ($colsA[$c] != $colsB[$c]) {
            $changed[] = $c;
        }
    }
    if ($colsAOnly || $colsBOnly || $changed) {
        echo "  * {$t}\n";
        foreach ($colsBOnly as $c) {
            echo "      + {$c} (only in B)\n";
        }
        foreach ($colsAOnly as $c) {
            echo "      - {$c} (only in A)\n";
        }
        foreach ($changed as $c) {
            $aDef = $colsA[$c];
            $bDef = $colsB[$c];
            echo "      ~ {$c} type/null/default differ:\n";
            echo "        A: ".json_encode($aDef)."\n";
            echo "        B: ".json_encode($bDef)."\n";
        }
    }
}

echo "\nDone.\n";
