<?php
// Repair core tables in bonny_delight when they exist "in metadata" but not in engine (error 1932)
// It only drops/recreates tables that are actually broken. Data in those tables will be lost.
// Usage: php scripts/repair_core_tables.php

error_reporting(E_ALL);
ini_set('display_errors', '1');

$host = getenv('BONNY_DB_HOST') ?: '127.0.0.1';
$user = getenv('BONNY_DB_USER') ?: 'root';
$pass = getenv('BONNY_DB_PASS') ?: 'vick247$';
$db   = getenv('BONNY_DB_NAME') ?: 'bonny_delight';

$core = ['users','categories','dishes','orders','order_items'];

try {
    $pdo = new PDO("mysql:host={$host};dbname={$db};charset=utf8mb4", $user, $pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
} catch (Throwable $e) {
    fwrite(STDERR, "Connection failed: {$e->getMessage()}\n");
    exit(1);
}

function table_broken(PDO $pdo, string $t): bool {
    try {
        $pdo->query("SELECT 1 FROM `{$t}` LIMIT 1");
        return false;
    } catch (PDOException $e) {
        $code = $e->getCode();
        // MySQL "doesn't exist in engine" 1932 or 42S02 errors
        if ($code == '42S02' || strpos($e->getMessage(), '1932') !== false) {
            return true;
        }
        throw $e; // unexpected; bubble up
    }
}

// Identify broken tables
$broken = [];
foreach ($core as $t) {
    try {
        if (table_broken($pdo, $t)) {
            $broken[] = $t;
        }
    } catch (Throwable $e) {
        // If SELECT fails with other errors, still consider broken to allow repair
        $broken[] = $t;
    }
}

if (!$broken) {
    echo "No broken core tables detected. Nothing to do.\n";
    exit(0);
}

echo "Broken tables: ".implode(', ', $broken)."\n";

// Drop in FK-safe order: order_items -> orders -> dishes -> categories -> users
$dropOrder = ['order_items','orders','dishes','categories','users'];
foreach ($dropOrder as $t) {
    if (in_array($t, $broken, true)) {
        echo "Dropping {$t}...\n";
        $pdo->exec("DROP TABLE IF EXISTS `{$t}`");
    }
}

// Recreate tables using definitions compatible with db.sql
// Users
if (in_array('users', $broken, true)) {
    echo "Creating users...\n";
    $pdo->exec(<<<SQL
CREATE TABLE IF NOT EXISTS users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(200) NOT NULL,
  first_name VARCHAR(100) NULL,
  last_name VARCHAR(100) NULL,
  gender ENUM('male','female') NULL,
  email VARCHAR(100) NOT NULL UNIQUE,
  phone VARCHAR(20) NOT NULL,
  password VARCHAR(255) NOT NULL,
  is_guest BOOLEAN DEFAULT 0,
  room_number VARCHAR(10),
  role ENUM('customer','admin','kitchen','rider') DEFAULT 'customer',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL);
}

// Categories
if (in_array('categories', $broken, true)) {
    echo "Creating categories...\n";
    $pdo->exec(<<<SQL
CREATE TABLE IF NOT EXISTS categories (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL);
}

// Dishes
if (in_array('dishes', $broken, true)) {
    echo "Creating dishes...\n";
    $pdo->exec(<<<SQL
CREATE TABLE IF NOT EXISTS dishes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  category_id INT,
  name VARCHAR(100) NOT NULL,
  description TEXT,
  price DECIMAL(10,2) NOT NULL,
  image VARCHAR(255),
  vegetarian BOOLEAN DEFAULT 0,
  halal BOOLEAN DEFAULT 1,
  gluten_free BOOLEAN DEFAULT 0,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_dish_category FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL);
}

// Orders
if (in_array('orders', $broken, true)) {
    echo "Creating orders...\n";
    $pdo->exec(<<<SQL
CREATE TABLE IF NOT EXISTS orders (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT,
  subtotal DECIMAL(10,2),
  delivery_fee DECIMAL(10,2),
  discount DECIMAL(10,2),
  total DECIMAL(10,2),
  status ENUM('Received','In Kitchen','Ready','Out for Delivery','Delivered') DEFAULT 'Received',
  delivery_address VARCHAR(255),
  payment_method ENUM('Cash on Delivery','Paystack','Charge to Room') DEFAULT 'Cash on Delivery',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_order_user FOREIGN KEY (user_id) REFERENCES users(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL);
}

// Order items
if (in_array('order_items', $broken, true)) {
    echo "Creating order_items...\n";
    $pdo->exec(<<<SQL
CREATE TABLE IF NOT EXISTS order_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  order_id INT,
  dish_id INT,
  quantity INT,
  price DECIMAL(10,2),
  CONSTRAINT fk_item_order FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
  CONSTRAINT fk_item_dish FOREIGN KEY (dish_id) REFERENCES dishes(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL);
}

echo "Core table repair complete.\n";
