<#
.SYNOPSIS
  Installs Node.js LTS, Python 3, Git, Composer (per-user), and the Laravel installer on Windows.

.DESCRIPTION
  Attempts installations with winget (per-user scope) first. If winget is missing or fails,
  falls back to Scoop (no admin required). Also sets PATH for Composer's global vendor/bin
  so the 'laravel' command is available. Optional flags install Yarn/pnpm and VS Build Tools.

.PARAMETER UseSystemScope
  Try system-wide installs with winget (may require Administrator). Default is per-user.

.PARAMETER WithYarnPnpm
  Install Yarn and pnpm globally via npm once Node.js is installed.

.PARAMETER WithVSBuildTools
  Attempt to install Visual Studio 2022 Build Tools for node-gyp and native addons.
  This often requires elevation and can be large; only use when needed.

.EXAMPLE
  # Per-user installs via winget or Scoop fallback
  ./install-dev-tools.ps1

.EXAMPLE
  # With Yarn/pnpm and VS Build Tools
  ./install-dev-tools.ps1 -WithYarnPnpm -WithVSBuildTools

.NOTES
  Re-open PowerShell after the script for PATH changes to take effect globally.
#>

param(
  [switch]$UseSystemScope,
  [switch]$WithYarnPnpm,
  [switch]$WithVSBuildTools
)

Set-StrictMode -Version Latest
$ErrorActionPreference = 'Stop'

function Write-Info($msg)  { Write-Host "[INFO]  $msg" -ForegroundColor Cyan }
function Write-Ok($msg)    { Write-Host "[OK]    $msg" -ForegroundColor Green }
function Write-Warn($msg)  { Write-Host "[WARN]  $msg" -ForegroundColor Yellow }
function Write-Err($msg)   { Write-Host "[ERROR] $msg" -ForegroundColor Red }

function Test-Command($name) {
  return [bool](Get-Command $name -ErrorAction SilentlyContinue)
}

function Add-ToCurrentPath([string]$path) {
  if (-not [string]::IsNullOrWhiteSpace($path) -and (Test-Path $path)) {
    if (-not ($env:Path -split ';' | Where-Object { $_.TrimEnd('\') -ieq $path.TrimEnd('\') })) {
      $env:Path += ";$path"
      Write-Info "Added to current session PATH: $path"
    }
  }
}

function Ensure-ComposerVendorBinOnPath() {
  $vendorBin = Join-Path $env:APPDATA 'Composer\vendor\bin'
  if (-not (Test-Path $vendorBin)) {
    New-Item -ItemType Directory -Path $vendorBin -Force | Out-Null
  }
  # Add for current session
  Add-ToCurrentPath $vendorBin
  # Persist for user
  $current = [Environment]::GetEnvironmentVariable('Path', 'User')
  if (-not ($current -split ';' | Where-Object { $_.TrimEnd('\') -ieq $vendorBin.TrimEnd('\') })) {
    [Environment]::SetEnvironmentVariable('Path', "$current;$vendorBin", 'User')
    Write-Info "Persisted to user PATH: $vendorBin (open a new PowerShell to use it)"
  }
}

function Install-WithWinget($id, [switch]$SystemScope) {
  if (-not (Test-Command winget)) { return $false }
  $scope = if ($SystemScope) { 'machine' } else { 'user' }
  $args = @('install','-e','--id', $id, '--accept-package-agreements','--accept-source-agreements','--scope', $scope)
  try {
    Write-Info "winget ${id} (scope=${scope})"
    $p = Start-Process -FilePath 'winget' -ArgumentList $args -WindowStyle Hidden -PassThru -Wait
    return ($p.ExitCode -eq 0)
  } catch {
    Write-Warn "winget failed for ${id}: $($_.Exception.Message)"
    return $false
  }
}

function Ensure-Scoop() {
  if (Test-Command scoop) { return $true }
  try {
    Write-Info 'Installing Scoop (user-space package manager)'
    # Ensure TLS 1.2
    [Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
    iwr -useb https://get.scoop.sh | iex
    return (Test-Command scoop)
  } catch {
    Write-Warn "Scoop install failed: $($_.Exception.Message)"
    return $false
  }
}

function Install-WithScoop($pkg) {
  if (-not (Ensure-Scoop)) { return $false }
  try {
    Write-Info "scoop install ${pkg}"
    if ($pkg -eq 'nodejs-lts') {
      scoop bucket add versions | Out-Null
    }
    scoop install $pkg
    return $true
  } catch {
    Write-Warn "scoop install failed for ${pkg}: $($_.Exception.Message)"
    return $false
  }
}

function Try-Install($name, $check, $wingetId, $scoopPkg) {
  if (& $check) { Write-Ok "$name already installed"; return $true }
  $ok = Install-WithWinget $wingetId -SystemScope:$UseSystemScope
  if (-not $ok) { $ok = Install-WithScoop $scoopPkg }
  if ($ok -and (& $check)) { Write-Ok "$name installed"; return $true }
  Write-Err "Failed to install $name with winget and scoop"
  return $false
}

# Pre-flight
Write-Info "Starting Windows dev tools install"
try { Set-ExecutionPolicy -Scope CurrentUser -ExecutionPolicy RemoteSigned -Force } catch { }

# 1) Git
$null = Try-Install 'Git' { Test-Command 'git' } 'Git.Git' 'git'

# 2) Node.js LTS (includes npm)
$null = Try-Install 'Node.js LTS' { Test-Command 'node' } 'OpenJS.NodeJS.LTS' 'nodejs-lts'

# 3) Python 3
$null = Try-Install 'Python 3' { Test-Command 'python' } 'Python.Python.3' 'python'

# 4) Composer
$composerOk = $false
if (Test-Command 'composer') {
  Write-Ok 'Composer already installed'
  $composerOk = $true
} else {
  $composerOk = Install-WithWinget 'Composer.Composer' -SystemScope:$UseSystemScope
  if (-not $composerOk) { $composerOk = Install-WithScoop 'composer' }
  if (-not $composerOk) {
    # Fallback to repo composer.phar using XAMPP PHP for global installs
    $php = 'C:\xampp\php\php.exe'
    $phar = Join-Path (Get-Location) 'composer.phar'
    if (Test-Path $php -PathType Leaf -and (Test-Path $phar -PathType Leaf)) {
      try {
        Write-Warn 'Using local composer.phar with XAMPP PHP as a fallback'
        & $php $phar --version | Out-Null
        $composerOk = $true
      } catch {
        Write-Warn 'composer.phar fallback failed'
      }
    }
  }
}

# 5) Laravel installer
if ($composerOk) {
  Ensure-ComposerVendorBinOnPath
  try {
    if (Test-Command 'composer') {
      Write-Info 'Installing laravel/installer globally (composer)'
      composer global require laravel/installer --no-interaction
    } else {
      $php = 'C:\xampp\php\php.exe'
      $phar = Join-Path (Get-Location) 'composer.phar'
      if (Test-Path $php -PathType Leaf -and (Test-Path $phar -PathType Leaf)) {
        Write-Info 'Installing laravel/installer globally via composer.phar'
        & $php $phar global require laravel/installer --no-interaction
      }
    }
  } catch {
    Write-Warn "Laravel installer installation failed: $($_.Exception.Message)"
  }
  # Add vendor/bin to PATH again in case it was created now
  Ensure-ComposerVendorBinOnPath
} else {
  Write-Warn 'Skipping Laravel installer because Composer is not available.'
}

# 6) Optional tools: Yarn & pnpm
if ($WithYarnPnpm -and (Test-Command 'npm')) {
  try {
    Write-Info 'Installing Yarn and pnpm globally via npm'
    npm install --location=global yarn pnpm --silent
  } catch {
    Write-Warn "Failed to install Yarn/pnpm: $($_.Exception.Message)"
  }
}

# 7) Optional: VS Build Tools
if ($WithVSBuildTools) {
  $ok = Install-WithWinget 'Microsoft.VisualStudio.2022.BuildTools' -SystemScope:$UseSystemScope
  if (-not $ok) { Write-Warn 'VS Build Tools installation may require Administrator. Please try running an elevated PowerShell.' }
}

# 8) Final status
Write-Host ''
Write-Info 'Final status:'
Write-Host ('Node.js:    ' + ($(if (Test-Command node) { node -v } else { 'Not found' })))
Write-Host ('npm:        ' + ($(if (Test-Command npm) { npm -v } else { 'Not found' })))
Write-Host ('Python:     ' + ($(if (Test-Command python) { python --version } else { 'Not found' })))
Write-Host ('py:         ' + ($(if (Test-Command py) { py --version } else { 'Not found' })))
Write-Host ('Git:        ' + ($(if (Test-Command git) { git --version } else { 'Not found' })))
Write-Host ('Composer:   ' + ($(if (Test-Command composer) { composer --version } else { 'Not found (using composer.phar fallback if available)' })))
Write-Host ('Laravel:    ' + ($(if (Test-Command laravel) { laravel --version } else { 'Not found' })))
Write-Host ''
Write-Ok 'Done. If commands are not recognized, open a new PowerShell window to refresh PATH.'
