# Runs the Laravel dev server using the port from .env
# Usage: powershell -ExecutionPolicy Bypass -File scripts\windows\serve.ps1

param(
    [switch]$AllInterfaces
)

$ErrorActionPreference = 'Stop'

function Get-ProjectRoot {
    # scripts/windows -> project root
    return (Resolve-Path (Join-Path $PSScriptRoot '..\..')).Path
}

function Get-PHPExe {
    $php = (Get-Command php -ErrorAction SilentlyContinue).Path
    if (-not $php) {
        $candidate = 'C:\\xampp\\php\\php.exe'
        if (Test-Path $candidate) { return $candidate }
    }
    return $php
}

function Read-EnvPort($envPath) {
    if (-not (Test-Path $envPath)) { return 8000 }
    $line = Select-String -Path $envPath -Pattern '^APP_PORT=' -SimpleMatch -ErrorAction SilentlyContinue | Select-Object -First 1
    if ($null -eq $line) { return 8000 }
    $val = ($line.Line -replace '^APP_PORT=', '').Trim()
    if ([int]::TryParse($val, [ref]([int]$null))) { return [int]$val }
    return 8000
}

function Stop-PhpServers([int]$port) {
    Get-CimInstance Win32_Process -Filter "Name='php.exe'" |
      Where-Object { $_.CommandLine -match "artisan serve" -or $_.CommandLine -match ":$port" -or $_.CommandLine -match "php\.exe -S" } |
      ForEach-Object { Stop-Process -Id $_.ProcessId -Force -ErrorAction SilentlyContinue }
}

$root = Get-ProjectRoot
Set-Location $root

$phpExe = Get-PHPExe
if (-not $phpExe) {
    Write-Error "PHP executable not found. Ensure PHP is installed or C:\\xampp\\php is available."
}

$envPath = Join-Path $root '.env'
$port = Read-EnvPort $envPath
$host = $AllInterfaces.IsPresent ? '0.0.0.0' : '127.0.0.1'

Write-Host "Using APP_PORT=$port HOST=$host"

# Clear caches (ignore warnings)
& $phpExe artisan config:clear 2>$null | Out-Null
& $phpExe artisan route:clear 2>$null | Out-Null
& $phpExe artisan view:clear 2>$null | Out-Null
& $phpExe artisan cache:clear 2>$null | Out-Null

# Stop any existing php servers on this port
Stop-PhpServers -port $port

# Start server
Write-Host "Starting Laravel dev server..."
Start-Process -FilePath $phpExe -ArgumentList @('artisan','serve',"--host=$host","--port=$port") -WindowStyle Normal

Start-Sleep -Seconds 2

# Probe server
try {
    $resp = Invoke-WebRequest -UseBasicParsing "http://127.0.0.1:$port/healthz" -TimeoutSec 5
    Write-Host "Health probe: $($resp.StatusCode)"
} catch {
    Write-Warning "Health probe failed: $($_.Exception.Message)"
}

# Open in default browser on localhost (for nicer URL)
Start-Process "http://localhost:$port"

Write-Host "Server should be running at http://localhost:$port"
