# Sets up bonny.local for Apache vhost dev on Windows (run as Administrator)
# - Adds hosts entry 127.0.0.1 bonny.local
# - Sets APP_URL in .env to http://bonny.local
# - Clears Laravel caches
# - Prints a ready-to-paste vhost block

param(
    [string]$ProjectRoot = "C:\\xampp\\htdocs\\bonny_delight",
    [string]$HostsPath = "C:\\Windows\\System32\\drivers\\etc\\hosts"
)

Write-Host "== Bonny Delight local setup =="

# 1) Hosts entry
$entry = "127.0.0.1`tbonny.local"
$hosts = Get-Content -Path $HostsPath -ErrorAction SilentlyContinue
if ($hosts -and ($hosts -join "`n") -match "\bbonny\.local\b") {
    Write-Host "Hosts entry already present." -ForegroundColor Green
} else {
    try {
        Add-Content -Path $HostsPath -Value $entry -ErrorAction Stop
        Write-Host "Added hosts entry: $entry" -ForegroundColor Green
    } catch {
        Write-Warning "Could not write to hosts file. Please run this script as Administrator."
    }
}

# 2) Set APP_URL in .env
$envPath = Join-Path $ProjectRoot ".env"
if (Test-Path $envPath) {
    $envText = Get-Content $envPath -Raw
    if ($envText -match "^APP_URL=.*") {
        $envText = [Regex]::Replace($envText, "^APP_URL=.*", "APP_URL=http://bonny.local", [System.Text.RegularExpressions.RegexOptions]::Multiline)
    } else {
        $envText = $envText.TrimEnd()+"`r`nAPP_URL=http://bonny.local`r`n"
    }
    $envText | Set-Content $envPath -Encoding UTF8
    Write-Host "Updated APP_URL in .env to http://bonny.local" -ForegroundColor Green
} else {
    Write-Warning ".env not found at $envPath"
}

# 3) Clear Laravel caches
$php = "C:\\xampp\\php\\php.exe"
Push-Location $ProjectRoot
try {
    & $php artisan optimize:clear | Out-Null
    Write-Host "Laravel caches cleared." -ForegroundColor Green
} catch {
    Write-Warning "Failed to clear caches. Ensure PHP is installed at $php"
}
Pop-Location

# 4) Print vhost block
$docRoot = Join-Path $ProjectRoot "public"
$vhost = @"
<VirtualHost *:80>
    ServerName bonny.local
    ServerAlias www.bonny.local
    DocumentRoot "$docRoot"

    <Directory "$docRoot">
        AllowOverride All
        Options Indexes FollowSymLinks
        Require all granted
    </Directory>

    ErrorLog "logs/bonny-error.log"
    CustomLog "logs/bonny-access.log" combined
</VirtualHost>
"@

Write-Host "`n== Paste this into C:/xampp/apache/conf/extra/httpd-vhosts.conf ==`n" -ForegroundColor Cyan
Write-Output $vhost
Write-Host "`nThen restart Apache from XAMPP Control Panel and browse http://bonny.local`n" -ForegroundColor Yellow
