<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Dish;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CheckoutAddressValidationTest extends TestCase
{
    // use RefreshDatabase; // Uncomment if migrations stable

    protected function setUp(): void
    {
        parent::setUp();
        \Illuminate\Support\Facades\Config::set('features.features.checkout', true); // ensure feature enabled for validation tests
    }

    protected function seedBasicDish(): Dish
    {
        return Dish::factory()->create(['price' => 1500, 'stock' => 10]);
    }

    public function test_external_delivery_requires_address()
    {
        $user = User::factory()->create();
        $dish = $this->seedBasicDish();

        $this->actingAs($user);
        session(['cart' => ['items' => [$dish->id => 1]]]);

        $resp = $this->post('/checkout', [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '',
        ]);
        $resp->assertSessionHasErrors(['delivery_address']);
    }

    public function test_room_delivery_requires_room_number()
    {
        $user = User::factory()->create(['room_number' => null]);
        $dish = $this->seedBasicDish();
        $this->actingAs($user);
        session(['cart' => ['items' => [$dish->id => 1]]]);

        // Force heuristic to choose room delivery by passing a room-like string but without actual number
        $resp = $this->post('/checkout', [
            'payment_method' => 'Charge to Room',
            'delivery_address' => '',
        ]);
        $resp->assertSessionHasErrors(['delivery_address']);
    }

    public function test_room_delivery_success_normalizes_address()
    {
        $user = User::factory()->create(['room_number' => '305', 'is_guest' => true]);
        $dish = $this->seedBasicDish();
        $this->actingAs($user);
        session(['cart' => ['items' => [$dish->id => 1]]]);

        $resp = $this->post('/checkout', [
            'payment_method' => 'Charge to Room',
            'delivery_mode' => 'room',
            'delivery_address' => '',
            'room_number' => '305',
        ]);
        
        $resp->assertRedirect();
        $this->assertDatabaseHas('orders', [ 'user_id' => $user->id, 'delivery_address' => 'Room 305' ]);
    }

    public function test_external_delivery_success()
    {
        $user = User::factory()->create();
        $dish = $this->seedBasicDish();
        $this->actingAs($user);
        session(['cart' => ['items' => [$dish->id => 1]]]);

        $resp = $this->post('/checkout', [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '42 Market Street',
        ]);
        $resp->assertRedirect();
        $this->assertDatabaseHas('orders', [ 'user_id' => $user->id, 'delivery_address' => '42 Market Street' ]);
    }
}
