<?php

namespace Tests\Feature;

use App\Models\Dish;
use App\Models\Order;
use App\Models\User;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Http;
use Tests\TestCase;

class CheckoutSuccessAndPaystackTest extends TestCase
{
    /**
     * Ensure a standard Cash on Delivery checkout succeeds, creates order + items, clears cart, and sets correct state.
     */
    public function test_checkout_cash_on_delivery_success_creates_order_and_clears_cart(): void
    {
        Config::set('features.features.checkout', true);

        $user = User::factory()->create();
        $dish = Dish::factory()->create(['price' => 2500]);

        $this->withSession(['cart' => ['items' => [$dish->id => 2]]])
            ->actingAs($user);

        $resp = $this->post('/checkout', [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '12 Palm Avenue',
        ]);

        $resp->assertRedirect();

        $order = Order::first();
        $this->assertNotNull($order, 'Order should be persisted');
        $this->assertEquals($user->id, $order->user_id);
        $this->assertEquals('12 Palm Avenue', $order->delivery_address);
        $this->assertEquals(2500 * 2, (int) $order->subtotal);
    $this->assertEquals(\App\Enums\OrderState::AWAITING_KITCHEN->value, $order->state);
        $this->assertDatabaseCount('order_items', 1);

        // Cart should be cleared after non-Paystack success
        $this->assertNull(session('cart'));
    }

    /**
     * Simulate Paystack initialization failure: order should roll back and error returned, cart retained.
     */
    public function test_checkout_paystack_initialization_failure_rolls_back_and_returns_error(): void
    {
        Config::set('features.features.checkout', true);
        Config::set('services.paystack.secret', 'sk_test_dummy');

        // Fake Paystack initialize endpoint returning status=false
        Http::fake([
            'api.paystack.co/transaction/initialize' => Http::response([
                'status' => false,
                'message' => 'Invalid authorization key',
            ], 200),
        ]);

        $user = User::factory()->create();
        $dish = Dish::factory()->create(['price' => 1800]);

        $this->withSession(['cart' => ['items' => [$dish->id => 1]]])
            ->actingAs($user);

        $resp = $this->post('/checkout', [
            'payment_method' => 'Paystack',
            'delivery_address' => '42 Market Street',
        ]);

        $resp->assertRedirect(); // back with errors
        $resp->assertSessionHasErrors(['checkout']);

    $this->assertSame(0, Order::count(), 'Order should not persist on Paystack failure');
        // Cart should still exist since payment not completed
        $this->assertNotNull(session('cart'), 'Cart retained after failure');
    }

    /**
     * Simulate a successful Paystack initialize call: expect redirect to authorization URL, order persisted,
     * payment reference stored, state awaiting_payment, cart retained (not cleared until payment success callback).
     */
    public function test_checkout_paystack_success_initializes_and_redirects(): void
    {
        Config::set('features.features.checkout', true);
        Config::set('services.paystack.secret', 'sk_test_dummy');

        $authUrl = 'https://paystack.mock/redirect/abc123';

        \Illuminate\Support\Facades\Http::fake([
            'api.paystack.co/transaction/initialize' => \Illuminate\Support\Facades\Http::response([
                'status' => true,
                'message' => 'Authorization URL created',
                'data' => [
                    'authorization_url' => $authUrl,
                    'access_code' => 'ACCESSCODE123',
                    'reference' => 'REFIGNORE', // controller generates its own reference
                ],
            ], 200),
        ]);

        $user = User::factory()->create();
        $dish = Dish::factory()->create(['price' => 3100, 'stock' => 6]);

        $this->withSession(['cart' => ['items' => [$dish->id => 2]]])
            ->actingAs($user);

        $resp = $this->post('/checkout', [
            'payment_method' => 'Paystack',
            'delivery_address' => '7 Success Way',
        ]);

        $resp->assertRedirect($authUrl);

        $order = Order::first();
        $this->assertNotNull($order, 'Order should persist on Paystack init success');
        $this->assertEquals($user->id, $order->user_id);
        $this->assertEquals('7 Success Way', $order->delivery_address);
    $this->assertEquals(\App\Enums\OrderState::AWAITING_PAYMENT->value, $order->state, 'Order state should be awaiting_payment');
        $this->assertNotEmpty($order->payment_reference, 'Payment reference stored');
        $this->assertMatchesRegularExpression('/^BD[0-9]{14}\d+$/', $order->payment_reference, 'Reference format BDYYYYmmddHHMMSS<ID>');

        // Cart should still exist (not cleared until payment completes)
        $this->assertNotNull(session('cart'));

        // Stock decremented and movement logged
        $this->assertSame(6 - 2, $dish->fresh()->stock);
        $this->assertDatabaseHas('order_items', [ 'order_id' => $order->id, 'dish_id' => $dish->id, 'quantity' => 2 ]);
        $this->assertTrue(\Illuminate\Support\Facades\DB::table('inventory_movements')->where('order_id', $order->id)->where('dish_id', $dish->id)->exists(), 'Inventory movement recorded');
    }
}
