<?php

namespace Tests\Feature\Inventory;

use Tests\TestCase;
use App\Models\User;
use App\Models\Dish;
use App\Models\Category;
use App\Models\Order;
use App\Models\InventoryMovement;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Config;

class CheckoutInventoryIntegrationTest extends TestCase
{
    use RefreshDatabase;

    private User $user;
    private Dish $dish;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Enable checkout feature for testing
        Config::set('features.features.checkout', true);
        
        // Disable middleware that might interfere with testing
        $this->withoutMiddleware([
            \App\Http\Middleware\VerifyCsrfToken::class,
            'orders.capacity'  // Disable capacity middleware for testing
        ]);
        
        // Create test user
        $this->user = User::factory()->create();
        
        // Create category and dish
        $category = Category::factory()->create();
        $this->dish = Dish::factory()->create([
            'category_id' => $category->id,
            'stock' => 10,
            'price' => 100.00,
            'name' => 'Test Dish'
        ]);
    }

    public function test_checkout_uses_inventory_service()
    {
        // Authenticate user
        $this->actingAs($this->user);
        
        // Set up cart in session
        Session::put('cart', [
            'items' => [
                $this->dish->id => 3
            ],
            'subtotal' => 300.00
        ]);

        // Perform checkout
        $response = $this->post(route('checkout.store'), [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '123 Test Street, Test City',
            'tip' => 0,
        ]);

        // Assert checkout was successful
        $response->assertRedirect();
        $response->assertSessionHas('status', 'Order placed');

        // Assert order was created
        $order = Order::where('user_id', $this->user->id)->first();
        $this->assertNotNull($order);

        // Assert stock was decremented using InventoryService
        $this->dish->refresh();
        $this->assertEquals(7, $this->dish->stock); // 10 - 3 = 7

        // Assert inventory movement was recorded
        $movement = InventoryMovement::where('dish_id', $this->dish->id)->first();
        $this->assertNotNull($movement);
        $this->assertEquals(-3, $movement->change);
        $this->assertEquals('sale', $movement->reason);
        $this->assertEquals($order->id, $movement->order_id);
        $this->assertEquals($this->user->id, $movement->user_id);
        $this->assertStringContainsString("Order #{$order->id} checkout", $movement->note);
    }

    public function test_checkout_fails_with_insufficient_stock()
    {
        // Authenticate user
        $this->actingAs($this->user);
        
        // Set up cart with more items than available stock
        Session::put('cart', [
            'items' => [
                $this->dish->id => 15 // More than available stock (10)
            ],
            'subtotal' => 1500.00
        ]);

        // Perform checkout
        $response = $this->post(route('checkout.store'), [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '123 Test Street, Test City',
            'tip' => 0,
        ]);

        // Assert checkout failed
        $response->assertRedirect();
        $response->assertSessionHasErrors('checkout');

        // Assert no order was created
        $this->assertDatabaseMissing('orders', [
            'user_id' => $this->user->id
        ]);

        // Assert stock was not changed
        $this->dish->refresh();
        $this->assertEquals(10, $this->dish->stock);

        // Assert no inventory movement was recorded
        $this->assertDatabaseMissing('inventory_movements', [
            'dish_id' => $this->dish->id
        ]);
    }

    public function test_checkout_with_multiple_dishes()
    {
        // Create another dish
        $dish2 = Dish::factory()->create([
            'category_id' => $this->dish->category_id,
            'stock' => 5,
            'price' => 150.00,
            'name' => 'Test Dish 2'
        ]);

        // Authenticate user
        $this->actingAs($this->user);
        
        // Set up cart with multiple dishes
        Session::put('cart', [
            'items' => [
                $this->dish->id => 2,
                $dish2->id => 3
            ],
            'subtotal' => 650.00 // (2 * 100) + (3 * 150)
        ]);

        // Perform checkout
        $response = $this->post(route('checkout.store'), [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '123 Test Street, Test City',
            'tip' => 0,
        ]);

        // Assert checkout was successful
        $response->assertRedirect();
        $response->assertSessionHas('status', 'Order placed');

        // Assert order was created
        $order = Order::where('user_id', $this->user->id)->first();
        $this->assertNotNull($order);

        // Assert stock was decremented for both dishes
        $this->dish->refresh();
        $dish2->refresh();
        $this->assertEquals(8, $this->dish->stock); // 10 - 2 = 8
        $this->assertEquals(2, $dish2->stock); // 5 - 3 = 2

        // Assert inventory movements were recorded for both dishes
        $movements = InventoryMovement::where('order_id', $order->id)->get();
        $this->assertCount(2, $movements);

        $movement1 = $movements->where('dish_id', $this->dish->id)->first();
        $movement2 = $movements->where('dish_id', $dish2->id)->first();

        $this->assertEquals(-2, $movement1->change);
        $this->assertEquals(-3, $movement2->change);
        $this->assertEquals('sale', $movement1->reason);
        $this->assertEquals('sale', $movement2->reason);
    }

    public function test_pre_checkout_validation_prevents_overselling()
    {
        // Authenticate user
        $this->actingAs($this->user);
        
        // Set up cart with exactly the available stock
        Session::put('cart', [
            'items' => [
                $this->dish->id => 10 // Exactly available stock
            ],
            'subtotal' => 1000.00
        ]);

        // First checkout should succeed
        $response = $this->post(route('checkout.store'), [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '123 Test Street, Test City',
            'tip' => 0,
        ]);

        $response->assertRedirect();
        $response->assertSessionHas('status', 'Order placed');

        // Stock should now be 0
        $this->dish->refresh();
        $this->assertEquals(0, $this->dish->stock);

        // Create a new user for second order
        $user2 = User::factory()->create();
        $this->actingAs($user2);

        // Set up cart for second user
        Session::put('cart', [
            'items' => [
                $this->dish->id => 1 // Should fail as stock is 0
            ],
            'subtotal' => 100.00
        ]);

        // Second checkout should fail during pre-checkout validation
        $response = $this->post(route('checkout.store'), [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '456 Another Street, Test City',
            'tip' => 0,
        ]);

        $response->assertRedirect();
        $response->assertSessionHasErrors('checkout');
        $this->assertStringContainsString('Not enough stock', session('errors')->first('checkout'));
    }
}