<?php

namespace Tests\Feature;

use App\Models\Dish;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\User;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Tests\TestCase;

class InventoryStockDecrementTest extends TestCase
{
    /**
     * Ensure stock is decremented and inventory_movements rows are recorded per ordered dish.
     */
    public function test_checkout_decrements_stock_and_records_movements(): void
    {
        Config::set('features.features.checkout', true);

        $user = User::factory()->create();
        $dishA = Dish::factory()->create(['price' => 1200, 'stock' => 10]);
        $dishB = Dish::factory()->create(['price' => 800, 'stock' => 7]);

        $qtyA = 3; $qtyB = 5; // within available stock

        $this->withSession(['cart' => ['items' => [
            $dishA->id => $qtyA,
            $dishB->id => $qtyB,
        ]]])->actingAs($user);

        $resp = $this->post('/checkout', [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '55 Test Lane',
        ]);

        $resp->assertRedirect();

        $order = Order::first();
        $this->assertNotNull($order, 'Order should have been created');
        $this->assertEquals($user->id, $order->user_id);

        $this->assertEquals(2, OrderItem::count(), 'Two order items expected');

        $this->assertSame(10 - $qtyA, $dishA->fresh()->stock, 'Dish A stock decremented');
        $this->assertSame(7 - $qtyB, $dishB->fresh()->stock, 'Dish B stock decremented');

        $movementA = DB::table('inventory_movements')->where('order_id', $order->id)->where('dish_id', $dishA->id)->first();
        $movementB = DB::table('inventory_movements')->where('order_id', $order->id)->where('dish_id', $dishB->id)->first();

        $this->assertNotNull($movementA, 'Movement row for dish A');
        $this->assertNotNull($movementB, 'Movement row for dish B');
        $this->assertEquals(-$qtyA, $movementA->change);
        $this->assertEquals(-$qtyB, $movementB->change);
        $this->assertEquals('sale', $movementA->reason);
        $this->assertEquals('sale', $movementB->reason);
    }

    /**
     * If requested quantity exceeds available stock, the order should not be created and stock remains unchanged.
     */
    public function test_checkout_fails_when_stock_insufficient_and_rolls_back(): void
    {
        Config::set('features.features.checkout', true);

        $user = User::factory()->create();
        $dish = Dish::factory()->create(['price' => 2000, 'stock' => 2]);

        $this->withSession(['cart' => ['items' => [ $dish->id => 5 ]]])
            ->actingAs($user);

        $resp = $this->post('/checkout', [
            'payment_method' => 'Cash on Delivery',
            'delivery_address' => '88 Failure Blvd',
        ]);

        $resp->assertRedirect();
        $resp->assertSessionHasErrors(['checkout']);

        $this->assertSame(0, Order::count(), 'No order persisted on stock failure');
        $this->assertSame(2, $dish->fresh()->stock, 'Stock unchanged after rollback');
        $this->assertSame(0, DB::table('inventory_movements')->count(), 'No inventory movement created');
    }
}
