<?php

namespace Tests\Feature;

use App\Models\Dish;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\User;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Tests\TestCase;

class PaystackWebhookTest extends TestCase
{
    /**
     * Test that a valid charge.success webhook updates the order (payment_verified + status) and is idempotent.
     */
    public function test_charge_success_updates_order_and_is_idempotent(): void
    {
        Config::set('features.features.checkout', true);
        Config::set('services.paystack.secret', 'sk_test_webhook');

        $user = User::factory()->create();
        $dish = Dish::factory()->create(['price' => 1500, 'stock' => 5]);

        // Manually create an awaiting_payment order as if initiated via Paystack
        $order = Order::create([
            'user_id' => $user->id,
            'subtotal' => 1500,
            'delivery_fee' => 0,
            'discount' => 0,
            'service_charge' => 0,
            'total' => 1500,
            'status' => 'Received', // initial display status
            'state' => \App\Enums\OrderState::AWAITING_PAYMENT->value,
            'delivery_mode' => 'external',
            'delivery_address' => 'Test Addr',
            'payment_method' => 'Paystack',
            'tip' => 0,
        ]);
        OrderItem::create([
            'order_id' => $order->id,
            'dish_id' => $dish->id,
            'quantity' => 1,
            'price' => 1500,
        ]);

        $eventId = 987654321;
        $reference = 'WEBHOOKREF123';
        $payload = [
            'event' => 'charge.success',
            'data' => [
                'id' => $eventId,
                'reference' => $reference,
                'status' => 'success',
                'amount' => 1500 * 100, // kobo
                'metadata' => [ 'order_id' => $order->id ],
            ],
        ];
        $json = json_encode($payload);
        $signature = hash_hmac('sha512', $json, 'sk_test_webhook');

        $resp = $this->withHeader('X-Paystack-Signature', $signature)
            ->postJson('/paystack/webhook', $payload);

        $resp->assertStatus(200)->assertSeeText('ok');

        $order->refresh();
        $this->assertTrue((bool) $order->payment_verified, 'Order marked payment_verified');
        $this->assertEquals('Received', $order->status, 'Status left as Received');

        // Second delivery of same event should be idempotent (cache prevents reprocessing)
        $resp2 = $this->withHeader('X-Paystack-Signature', $signature)
            ->postJson('/paystack/webhook', $payload);
        $resp2->assertStatus(200)->assertSeeText('Already processed');
        $after = $order->fresh();
        $this->assertTrue((bool) $after->payment_verified);
    }

    /**
     * Invalid signature should yield 401 and not modify order.
     */
    public function test_invalid_signature_rejected(): void
    {
        Config::set('services.paystack.secret', 'sk_test_webhook');

        $user = User::factory()->create();
        $order = Order::create([
            'user_id' => $user->id,
            'subtotal' => 1000,
            'total' => 1000,
            'status' => 'Received',
            'state' => \App\Enums\OrderState::AWAITING_PAYMENT->value,
            'delivery_mode' => 'external',
            'delivery_address' => 'Somewhere',
            'payment_method' => 'Paystack',
        ]);

        $payload = [ 'event' => 'charge.success', 'data' => [ 'id' => 111, 'reference' => 'REF', 'status' => 'success', 'amount' => 100000, 'metadata' => ['order_id' => $order->id] ] ];

        $resp = $this->withHeader('X-Paystack-Signature', 'bogus')
            ->postJson('/paystack/webhook', $payload);

        $resp->assertStatus(401);
        $this->assertFalse((bool) $order->fresh()->payment_verified, 'Order should remain unverified');
    }
}
