<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use App\Services\Reporting\ReportingService;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Dish;
use App\Models\Category;
use App\Models\User;
use App\Models\InventoryMovement;
use Carbon\Carbon;

class ReportingServiceTest extends TestCase
{
    use RefreshDatabase;

    private ReportingService $reportingService;
    private User $user;
    private Category $category;
    private Dish $dish1;
    private Dish $dish2;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->reportingService = new ReportingService();
        
        // Create test data
        $this->user = User::factory()->create();
        $this->category = Category::factory()->create(['name' => 'Test Category']);
        
        $this->dish1 = Dish::factory()->create([
            'name' => 'Test Dish 1',
            'price' => 1000.00,
            'stock' => 50,
            'low_stock_threshold' => 5,
            'category_id' => $this->category->id,
        ]);
        
        $this->dish2 = Dish::factory()->create([
            'name' => 'Test Dish 2',
            'price' => 1500.00,
            'stock' => 30,
            'low_stock_threshold' => 3,
            'category_id' => $this->category->id,
        ]);
    }

    public function test_gross_margin_calculation()
    {
        $netSales = 10000;
        $cogs = 4000;
        
        $grossMargin = $this->reportingService->calculateGrossMargin($netSales, $cogs);
        
        // Expected: (10000 - 4000) / 10000 * 100 = 60%
        $this->assertEquals(60.0, $grossMargin);
    }

    public function test_gross_margin_with_zero_sales()
    {
        $grossMargin = $this->reportingService->calculateGrossMargin(0, 1000);
        $this->assertEquals(0.0, $grossMargin);
    }

    public function test_kpi_report_with_orders()
    {
        // Clear existing data for clean test
        Order::query()->delete();
        OrderItem::query()->delete();
        InventoryMovement::query()->delete();
        
        // Create test orders
        $order1 = Order::factory()->create([
            'user_id' => $this->user->id,
            'status' => 'Delivered',
            'subtotal' => 2500,
            'discount' => 100,
            'total' => 2400,
            'created_at' => now(),
        ]);

        $order2 = Order::factory()->create([
            'user_id' => $this->user->id,
            'status' => 'Delivered',
            'subtotal' => 1500,
            'discount' => 0,
            'total' => 1500,
            'created_at' => now(),
        ]);

        // Create order items manually
        OrderItem::create([
            'order_id' => $order1->id,
            'dish_id' => $this->dish1->id,
            'quantity' => 2,
            'price' => 1000.00,
        ]);

        OrderItem::create([
            'order_id' => $order1->id,
            'dish_id' => $this->dish2->id,
            'quantity' => 1,
            'price' => 1500.00,
        ]);

        OrderItem::create([
            'order_id' => $order2->id,
            'dish_id' => $this->dish1->id,
            'quantity' => 1,
            'price' => 1000.00,
        ]);

        // Create corresponding inventory movements
        InventoryMovement::create([
            'dish_id' => $this->dish1->id,
            'change' => -2,
            'reason' => 'sale',
            'user_id' => $this->user->id,
            'order_id' => $order1->id,
            'created_at' => now(),
        ]);

        InventoryMovement::create([
            'dish_id' => $this->dish2->id,
            'change' => -1,
            'reason' => 'sale',
            'user_id' => $this->user->id,
            'order_id' => $order1->id,
            'created_at' => now(),
        ]);

        InventoryMovement::create([
            'dish_id' => $this->dish1->id,
            'change' => -1,
            'reason' => 'sale',
            'user_id' => $this->user->id,
            'order_id' => $order2->id,
            'created_at' => now(),
        ]);

        // Generate report
        $report = $this->reportingService->generateKPIReport('daily');

        // Verify sales metrics
        $this->assertEquals(4000.0, $report['sales_metrics']['gross_sales']); // 2*1000 + 1*1500 + 1*1000
        $this->assertEquals(100.0, $report['sales_metrics']['discounts']);
        $this->assertEquals(3900.0, $report['sales_metrics']['net_sales']); // 4000 - 100
        $this->assertEquals(2, $report['sales_metrics']['total_orders']);
        $this->assertEquals(4, $report['sales_metrics']['total_items_sold']); // 2 + 1 + 1
        $this->assertEquals(1950.0, $report['sales_metrics']['average_order_value']); // 3900 / 2

        // Verify operational metrics
        $this->assertEquals(2, $report['operational_metrics']['delivered_orders']);
        $this->assertEquals(100.0, $report['operational_metrics']['fill_rate_percent']);
    }

    public function test_top_selling_dishes()
    {
        // Create inventory movements for sales
        InventoryMovement::create([
            'dish_id' => $this->dish1->id,
            'change' => -5,
            'reason' => 'sale',
            'user_id' => $this->user->id,
            'created_at' => now(),
        ]);

        InventoryMovement::create([
            'dish_id' => $this->dish2->id,
            'change' => -3,
            'reason' => 'sale',
            'user_id' => $this->user->id,
            'created_at' => now(),
        ]);

        InventoryMovement::create([
            'dish_id' => $this->dish1->id,
            'change' => -2,
            'reason' => 'sale',
            'user_id' => $this->user->id,
            'created_at' => now(),
        ]);

        $topSellers = $this->reportingService->getTopSellingDishes('daily');

        $this->assertCount(2, $topSellers);
        
        // Dish 1 should be top seller with 7 units sold
        $this->assertEquals($this->dish1->id, $topSellers[0]['dish_id']);
        $this->assertEquals('Test Dish 1', $topSellers[0]['dish_name']);
        $this->assertEquals(7, $topSellers[0]['total_sold']);
        $this->assertEquals(7000.0, $topSellers[0]['revenue']); // 7 * 1000

        // Dish 2 should be second with 3 units sold
        $this->assertEquals($this->dish2->id, $topSellers[1]['dish_id']);
        $this->assertEquals('Test Dish 2', $topSellers[1]['dish_name']);
        $this->assertEquals(3, $topSellers[1]['total_sold']);
        $this->assertEquals(4500.0, $topSellers[1]['revenue']); // 3 * 1500
    }

    public function test_reporting_with_custom_date_range()
    {
        // Clear existing data
        Order::query()->delete();
        OrderItem::query()->delete();
        InventoryMovement::query()->delete();
        
        $startDate = Carbon::now()->subDays(7);
        $endDate = Carbon::now()->subDays(1);

        // Create order in the past
        $order = Order::factory()->create([
            'user_id' => $this->user->id,
            'status' => 'Delivered',
            'total' => 1000,
            'created_at' => $startDate,
        ]);

        OrderItem::create([
            'order_id' => $order->id,
            'dish_id' => $this->dish1->id,
            'quantity' => 1,
            'price' => 1000.00,
        ]);

        InventoryMovement::create([
            'dish_id' => $this->dish1->id,
            'change' => -1,
            'reason' => 'sale',
            'user_id' => $this->user->id,
            'order_id' => $order->id,
            'created_at' => $startDate,
        ]);

        $report = $this->reportingService->generateKPIReport('weekly', $startDate, $endDate);

        $this->assertEquals($startDate->toDateString(), $report['period']['start_date']);
        $this->assertEquals($endDate->toDateString(), $report['period']['end_date']);
        $this->assertEquals(1000.0, $report['sales_metrics']['gross_sales']);
        $this->assertEquals(1, $report['sales_metrics']['total_orders']);
    }

    public function test_inventory_metrics_calculation()
    {
        // Clear existing dishes and create fresh ones for clean test
        Dish::query()->delete();
        
        $dish1 = Dish::factory()->create([
            'name' => 'Low Stock Dish',
            'price' => 1000.00,
            'stock' => 10,
            'low_stock_threshold' => 15, // This will be low stock
            'category_id' => $this->category->id,
        ]);
        
        $dish2 = Dish::factory()->create([
            'name' => 'Out of Stock Dish',
            'price' => 1500.00,
            'stock' => 0, // This will be out of stock
            'low_stock_threshold' => 5,
            'category_id' => $this->category->id,
        ]);

        $report = $this->reportingService->generateKPIReport('daily');

        $this->assertGreaterThan(0, $report['inventory_metrics']['current_inventory_value']);
        $this->assertEquals(10, $report['inventory_metrics']['total_on_hand']); // 10 + 0
        $this->assertEquals(1, $report['inventory_metrics']['low_stock_count']); // dish1 is below threshold
        $this->assertEquals(1, $report['inventory_metrics']['out_of_stock_count']); // dish2 is out of stock
    }
}
