<?php

namespace Tests\Unit;

use App\Enums\OrderState;
use App\Events\OrderStateChanged;
use App\Models\Order;
use App\Support\Orders\OrderLifecycle;
use Illuminate\Support\Facades\Event;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OrderLifecycleTest extends TestCase
{
    use RefreshDatabase;

    protected OrderLifecycle $lifecycle;

    protected function setUp(): void
    {
        parent::setUp();
        $this->lifecycle = new OrderLifecycle();
    }

    public static function allowedTransitionsProvider(): array
    {
        return [
            [OrderState::AWAITING_PAYMENT, OrderState::AWAITING_KITCHEN],
            [OrderState::AWAITING_PAYMENT, OrderState::AWAITING_REVIEW],
            [OrderState::AWAITING_KITCHEN, OrderState::PREPARING],
            [OrderState::PREPARING, OrderState::READY],
            [OrderState::READY, OrderState::DELIVERING],
            [OrderState::DELIVERING, OrderState::COMPLETED],
        ];
    }

    /**
     * @dataProvider allowedTransitionsProvider
     */
    public function test_allowed_transitions_emit_event(OrderState $from, OrderState $to): void
    {
        Event::fake();
        $order = Order::factory()->state(['state' => $from->value, 'status' => $from->toDisplayStatus()])->create();

        $ok = $this->lifecycle->transition($order, $to);
        $this->assertTrue($ok, 'Transition should succeed');
        $this->assertEquals($to->value, $order->fresh()->state);
        $this->assertEquals($to->toDisplayStatus(), $order->fresh()->status);
        Event::assertDispatched(OrderStateChanged::class, function($e) use ($order, $from, $to) {
            return $e->order->id === $order->id && $e->from === $from && $e->to === $to;
        });
    }

    public static function forbiddenTransitionsProvider(): array
    {
        return [
            [OrderState::COMPLETED, OrderState::PREPARING],
            [OrderState::CANCELLED, OrderState::READY],
            [OrderState::AWAITING_PAYMENT, OrderState::DELIVERING], // skipping path
        ];
    }

    /**
     * @dataProvider forbiddenTransitionsProvider
     */
    public function test_forbidden_transitions_do_not_change_state(OrderState $from, OrderState $to): void
    {
        Event::fake();
        $order = Order::factory()->state(['state' => $from->value, 'status' => $from->toDisplayStatus()])->create();
        $result = $this->lifecycle->transition($order, $to);
        $this->assertFalse($result);
        $this->assertEquals($from->value, $order->fresh()->state);
        Event::assertNotDispatched(OrderStateChanged::class);
    }

    public function test_same_state_is_noop(): void
    {
        Event::fake();
        $order = Order::factory()->preparing()->create();
        $ok = $this->lifecycle->transition($order, OrderState::PREPARING);
        $this->assertFalse($ok);
        $this->assertEquals(OrderState::PREPARING->value, $order->fresh()->state);
        Event::assertNotDispatched(OrderStateChanged::class);
    }

    public function test_all_enum_values_are_covered_by_tests(): void
    {
        $all = collect(OrderState::cases())->map(fn($c) => $c->value)->all();
        $mentioned = [
            // From allowed
            OrderState::AWAITING_PAYMENT->value,
            OrderState::AWAITING_KITCHEN->value,
            OrderState::AWAITING_REVIEW->value,
            OrderState::PREPARING->value,
            OrderState::READY->value,
            OrderState::DELIVERING->value,
            OrderState::COMPLETED->value,
            OrderState::CANCELLED->value,
        ];
        sort($all); sort($mentioned);
        $this->assertSame($all, $mentioned, 'Enum states missing in test coverage list');
    }
}
