<?php

namespace Tests\Unit\Services\Inventory;

use Tests\TestCase;
use App\Services\Inventory\InventoryService;
use App\Models\Dish;
use App\Models\Order;
use App\Models\User;
use App\Models\InventoryMovement;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\DB;

class InventoryServiceTest extends TestCase
{
    use RefreshDatabase;

    private InventoryService $inventoryService;
    private Dish $dish;
    private User $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->inventoryService = app(InventoryService::class);
        
        // Create test user
        $this->user = User::factory()->create();
        
        // Create test dish with initial stock
        $this->dish = Dish::factory()->create([
            'stock' => 10,
            'name' => 'Test Dish',
            'price' => 100.00
        ]);
    }

    public function test_record_sale_success()
    {
        $order = Order::factory()->create();
        $quantity = 3;
        
        $movement = $this->inventoryService->recordSale(
            dish: $this->dish,
            quantity: $quantity,
            order: $order,
            user: $this->user,
            note: 'Test sale'
        );

        // Assert movement was created correctly
        $this->assertInstanceOf(InventoryMovement::class, $movement);
        $this->assertEquals($this->dish->id, $movement->dish_id);
        $this->assertEquals(-$quantity, $movement->change);
        $this->assertEquals('sale', $movement->reason);
        $this->assertEquals($order->id, $movement->order_id);
        $this->assertEquals($this->user->id, $movement->user_id);
        $this->assertEquals('Test sale', $movement->note);

        // Assert stock was decremented
        $this->dish->refresh();
        $this->assertEquals(7, $this->dish->stock);
    }

    public function test_record_sale_insufficient_stock()
    {
        $order = Order::factory()->create();
        
        $this->expectException(ValidationException::class);
        $this->expectExceptionMessage('Insufficient stock');
        
        $this->inventoryService->recordSale(
            dish: $this->dish,
            quantity: 15, // More than available stock (10)
            order: $order,
            user: $this->user
        );

        // Assert stock was not changed
        $this->dish->refresh();
        $this->assertEquals(10, $this->dish->stock);
    }

    public function test_record_sale_invalid_quantity()
    {
        $order = Order::factory()->create();
        
        $this->expectException(ValidationException::class);
        $this->expectExceptionMessage('Sale quantity must be positive');
        
        $this->inventoryService->recordSale(
            dish: $this->dish,
            quantity: 0,
            order: $order,
            user: $this->user
        );
    }

    public function test_adjust_increase_stock()
    {
        $movement = $this->inventoryService->adjust(
            dish: $this->dish,
            change: 5,
            reason: 'adjustment',
            user: $this->user,
            note: 'Stock replenishment'
        );

        // Assert movement was created
        $this->assertEquals(5, $movement->change);
        $this->assertEquals('adjustment', $movement->reason);
        $this->assertEquals('Stock replenishment', $movement->note);

        // Assert stock was increased
        $this->dish->refresh();
        $this->assertEquals(15, $this->dish->stock);
    }

    public function test_adjust_decrease_stock()
    {
        $movement = $this->inventoryService->adjust(
            dish: $this->dish,
            change: -3,
            reason: 'spoilage',
            user: $this->user,
            note: 'Spoiled items removed'
        );

        // Assert movement was created
        $this->assertEquals(-3, $movement->change);
        $this->assertEquals('spoilage', $movement->reason);

        // Assert stock was decreased
        $this->dish->refresh();
        $this->assertEquals(7, $this->dish->stock);
    }

    public function test_adjust_would_make_stock_negative()
    {
        $this->expectException(ValidationException::class);
        $this->expectExceptionMessage('Insufficient stock');
        
        $this->inventoryService->adjust(
            dish: $this->dish,
            change: -15, // Would make stock negative
            reason: 'adjustment',
            user: $this->user
        );

        // Assert stock was not changed
        $this->dish->refresh();
        $this->assertEquals(10, $this->dish->stock);
    }

    public function test_adjust_zero_change()
    {
        $this->expectException(ValidationException::class);
        $this->expectExceptionMessage('Stock change cannot be zero');
        
        $this->inventoryService->adjust(
            dish: $this->dish,
            change: 0,
            reason: 'adjustment',
            user: $this->user
        );
    }

    public function test_invalid_movement_reason()
    {
        $this->expectException(ValidationException::class);
        $this->expectExceptionMessage('Invalid movement reason: invalid_reason');
        
        $this->inventoryService->adjust(
            dish: $this->dish,
            change: 5,
            reason: 'invalid_reason',
            user: $this->user
        );
    }

    public function test_get_available_stock()
    {
        $availableStock = $this->inventoryService->getAvailableStock($this->dish);
        $this->assertEquals(10, $availableStock);

        // Test with dish having zero stock
        $dishWithZeroStock = Dish::factory()->create(['stock' => 0]);
        $availableStock = $this->inventoryService->getAvailableStock($dishWithZeroStock);
        $this->assertEquals(0, $availableStock);
    }

    public function test_is_available()
    {
        $this->assertTrue($this->inventoryService->isAvailable($this->dish, 5));
        $this->assertTrue($this->inventoryService->isAvailable($this->dish, 10));
        $this->assertFalse($this->inventoryService->isAvailable($this->dish, 15));
    }

    public function test_movement_reason_validation()
    {
        $this->assertTrue($this->inventoryService->isValidReason('sale'));
        $this->assertTrue($this->inventoryService->isValidReason('adjustment'));
        $this->assertTrue($this->inventoryService->isValidReason('correction'));
        $this->assertFalse($this->inventoryService->isValidReason('invalid_reason'));
    }

    public function test_get_valid_reasons()
    {
        $reasons = $this->inventoryService->getValidReasons();
        
        $this->assertIsArray($reasons);
        $this->assertArrayHasKey('sale', $reasons);
        $this->assertArrayHasKey('adjustment', $reasons);
        $this->assertArrayHasKey('correction', $reasons);
        $this->assertEquals('Sale/Order fulfillment', $reasons['sale']);
    }

    public function test_concurrent_stock_operations_with_locking()
    {
        // This test simulates concurrent operations to ensure pessimistic locking works
        $dish = Dish::factory()->create(['stock' => 5]);
        
        // First transaction: should succeed
        DB::transaction(function () use ($dish) {
            $this->inventoryService->recordSale(
                dish: $dish,
                quantity: 3,
                user: $this->user
            );
        });

        // Second transaction: should fail due to insufficient stock
        $this->expectException(ValidationException::class);
        
        DB::transaction(function () use ($dish) {
            $this->inventoryService->recordSale(
                dish: $dish,
                quantity: 4, // Only 2 left after first transaction
                user: $this->user
            );
        });
    }

    public function test_movement_creates_proper_audit_trail()
    {
        $order = Order::factory()->create();
        
        // Perform multiple operations
        $this->inventoryService->recordSale($this->dish, 2, $order, $this->user, 'Sale 1');
        $this->inventoryService->adjust($this->dish, 5, 'adjustment', $this->user, 'Restock');
        $this->inventoryService->recordSale($this->dish, 1, null, $this->user, 'Sale 2');

        // Check audit trail
        $movements = InventoryMovement::where('dish_id', $this->dish->id)
            ->orderBy('created_at')
            ->get();

        $this->assertCount(3, $movements);
        
        // First movement: sale of 2
        $this->assertEquals(-2, $movements[0]->change);
        $this->assertEquals('sale', $movements[0]->reason);
        $this->assertEquals($order->id, $movements[0]->order_id);
        
        // Second movement: adjustment of +5
        $this->assertEquals(5, $movements[1]->change);
        $this->assertEquals('adjustment', $movements[1]->reason);
        $this->assertNull($movements[1]->order_id);
        
        // Third movement: sale of 1
        $this->assertEquals(-1, $movements[2]->change);
        $this->assertEquals('sale', $movements[2]->reason);
        $this->assertNull($movements[2]->order_id);

        // Final stock should be: 10 - 2 + 5 - 1 = 12
        $this->dish->refresh();
        $this->assertEquals(12, $this->dish->stock);
    }
}